/**
 * @file  sci.c
 * @brief SCIドライバ
 *
 * @version: 1.9 2024/10/17, $Id $
 *
 * 本ソフトウェアの著作権は作成元である(株)北斗電子が所有するものとし、
 * (株)北斗電子は、以下の (1)-(3) の条件を満たす場合に限り、
 * 本ソフトウェア（本ソフトウェアを改変したものを含む。以下同じ）を
 * 使用・複製・改変・再配布（以下、利用と呼ぶ）することを無償で許諾する。
 *
 * (1) 本ソフトウェアをソースコードの形で利用する場合には、下記の著作
 *	   権表示、この利用条件が、そのままの形でソースコード中に含まれて
 *	   いること。
 * (2) 本ソフトウェアの一部または全てを無断で転載することを禁止するもの
 *	   とする。雑誌などへ紹介・収録の場合は(株)北斗電子に連絡願います。
 * (3) 本ソフトウェアの利用により直接的または間接的に生じるいかなる損害
 *	   からも、(株)北斗電子は一切の責任を負わないものとする。
 *
 * Copyright (C) Hokuto denshi Co,Ltd. 2021-2024
 */

/*----------------------------------------------------------------------
	インクルード
----------------------------------------------------------------------*/
#include	"sci.h"

/*----------------------------------------------------------------------
	プロトタイプ宣言
----------------------------------------------------------------------*/
void sci_start(void);
void sci_stop(void);
#ifdef SCI_USE_TX
void sci_write_char(unsigned char c);
void sci_write_uint8_hex(unsigned char c);
void sci_write_uint16_hex(unsigned short s);
void sci_write_uint32_hex(unsigned long l);
void sci_write_uint8(unsigned char num);
void sci_write_uint16(unsigned short num);
void sci_write_uint32(unsigned long num);
void sci_write_int8(char num);
void sci_write_int16(short num);
void sci_write_int32(long num);
void sci_write_str(const char *str);
void sci_write_flush(void);
#endif
#ifdef SCI_USE_RX
unsigned short sci_read_char(unsigned char *c);
unsigned short sci_read_str(char *str, unsigned short size);
unsigned short sci_read_data_size(void);
void sci_read_buf_clear(void);
#endif
#ifdef SCI_USE_FLOAT
int float2str(float value, int num, char *str);
int float2str_eformat(float value, int num, char *str);
int double2str(double value, int num, char *str);
int double2str_eformat(double value, int num, char *str);
#endif

#ifdef SCI_USE_TX
void intr_sci_send_end(void);
#endif
#ifdef SCI_USE_RX
void intr_sci_receive_end(void);
void intr_sci_receive_error(void);
#endif

static int interrupt_flag_get(void);

#if defined(MCU_TYPE_RA)
//FSPで設定する割り込みコールバック関数
//※FSPのUARTのcallbackの名称を"user_uart_callback"に設定してください
void user_uart_callback (uart_callback_args_t * p_args);
#endif

/*----------------------------------------------------------------------
	グローバル変数（ファイル内）
----------------------------------------------------------------------*/
#ifdef SCI_USE_TX
static const unsigned char hex_table_char[] = "0123456789ABCDEF";		// 文字列テーブル
//static const unsigned char hex_table_char[] = "0123456789abcdef";		// 文字列テーブル（小文字表示）
#endif

/*----------------------------------------------------------------------
	グローバル変数
----------------------------------------------------------------------*/
#ifdef SCI_USE_TX
unsigned char g_sci_send_buf[2][SCI_SEND_BUF_SIZE];			//送信バッファ
#endif
#ifdef SCI_USE_RX
unsigned char g_sci_recv_buf[SCI_RECV_BUF_SIZE];			//受信バッファ
#endif

#ifdef SCI_USE_TX
volatile unsigned short g_sci_send_buf_index[2] = {0, 0};	//格納済みインデックス（送信）
volatile unsigned char g_sci_send_buf_page_index = 0;		//格納ページを示すインデックス（送信）
volatile unsigned char g_sci_send_flag = FLAG_CLEAR;		//送信中フラグ（送信）
volatile unsigned char g_sci_send_nowait_flag = FLAG_CLEAR;	//送信ウェイトを行わないフラグ（送信）
#endif
#ifdef SCI_USE_RX
volatile unsigned short g_sci_recv_buf_index1 = 0;			//格納済みインデックス（受信）
volatile unsigned short g_sci_recv_buf_index2 = 0;			//読み出し済みインデックス（受信)
unsigned char g_sci_recv_data;								//受信データ（受信）
#endif

volatile unsigned char g_sci_status = 0;					//ステータス
volatile unsigned char g_sci_start = FLAG_CLEAR;			//モジュール動作ステータス


/*----------------------------------------------------------------------
	ソースコード
----------------------------------------------------------------------*/

void sci_start( void )
{
	//SCI動作開始関数
	
	if (FLAG_SET == g_sci_start)
	{
		//RAでDTCを有効にした場合SCI_STARTの前にSCI_STOPを実行した場合エラーハンドラに飛ぶので
		//現在動作中か確認して停止する様にする
		
		SCI_STOP;
	}
	
#ifdef SCI_USE_TX
	g_sci_send_buf_index[0] = 0;
	g_sci_send_buf_index[1] = 0;
	g_sci_send_buf_page_index = 0;
	g_sci_send_flag = FLAG_CLEAR;
	g_sci_send_nowait_flag = FLAG_CLEAR;
#endif
#ifdef SCI_USE_RX
	g_sci_recv_buf_index1 = 0;
	g_sci_recv_buf_index2 = 0;
#endif
	g_sci_status = 0;
	
	SCI_START;
	g_sci_start = FLAG_SET;
	
#ifdef SCI_USE_RX
	//1文字受信処理
	SCI_RECEIVE(&g_sci_recv_data, 1);
#endif
}

void sci_stop( void )
{
	//SCI停止関数
	
	if (FLAG_SET == g_sci_start)
	{
		//RAでDTCを有効にした場合SCI_STARTの前にSCI_STOPを実行した場合エラーハンドラに飛ぶので
		//現在動作中か確認して停止する様にする
		
		SCI_STOP;
		g_sci_start = FLAG_CLEAR;
	}
}

#ifdef SCI_USE_TX
void sci_write_char( unsigned char c )
{
	//1文字送信関数（送信バッファ格納関数）
	
	//与えられた文字を送信バッファに格納する
	
	//引数
	// unsigned char c : 文字コード

	int i_flag = interrupt_flag_get();
	
	while (g_sci_send_buf_index[g_sci_send_buf_page_index] >= SCI_SEND_BUF_SIZE)
	{
		//バッファフルが解消されるまでループ
		
		//割り込み禁止状態、かつバッファフルで本関数を呼び出すと無限ループとなるので
		//その場合はバッファフル解消を待たずデータを捨ててリターンする
		
		if ((g_sci_send_nowait_flag == FLAG_SET) || (i_flag == 0))
		{
			g_sci_status |= SCI_SEND_BUF_OVERFLOW;
			
			return;		//nowaitフラグが設定されている時、または割込み禁止時はデータを格納せずに捨てる
		}
		
		//whileループに入った後で割込み禁止になった場合、無限ループになるので
		//割込み禁止・許可フラグは随時チェックする
		i_flag = interrupt_flag_get();
	}
	
	//送信バッファとインデックスに不整合があると表示がおかしくなるので、送信バッファ格納とインデックスインクリメント間の割り込みを禁止する
	if(i_flag) INTERRUPT_DISABLE;	//割込みフラグにより処理内容を変える必要性はないが、元々割込み禁止の時は何もしない
	g_sci_send_buf[g_sci_send_buf_page_index][g_sci_send_buf_index[g_sci_send_buf_page_index]++] = c;	//送信バッファに格納 & インデックスをインクリメント
	if(i_flag) INTERRUPT_ENABLE;	//割り込み禁止状態で本関数が呼ばれた場合は割り込みを許可しない
	
	NOP_MCU_DEP;		//割り込み禁止中に割り込みフラグが立った際はこのNOP命令の実行後、割り込み処理に飛ぶ(RL78)
	
	if (g_sci_send_flag == FLAG_CLEAR)   //送信中ではない
	{
		g_sci_send_flag = FLAG_SET;
		
		//送信関数をキックする
		
		if(i_flag) INTERRUPT_DISABLE;
		
		//基本的には SCI_SEND() 実行から送信終了割り込みまではある程度時間的な猶予があるので
		//割り込み禁止とする必要性は薄いが念のため割り込み禁止設定内で処理を行う
		
#if defined(MCU_TYPE_RH850)
		
		//-RH850での送信ライブラリ関数呼び出し時の注意点-
		
		//RH850はマイコンの機能としては送信完了割り込みを掛ける事が可能であるが
		//「送信エンプティ割り込み」と「送信完了割り込み」が分かれていない
		
		//スマートコンフィグレータの生成コードは送信バッファが空になったタイミングでの割り込みとなる
		//→「送信エンプティ割り込み」固定のコード
		
		//送信中に SCI_SEND を呼び出すと MD_ERROR を返すので、そのときは再度 SCI_SEND を呼び出す
		//→最大1キャラクタの送信時間のウェイトとなるので、効率が悪い
		//→最終データ送信時に、送信割り込みの設定を変更したいが、スマートコンフィグレータ内のコード変更要となるため
		//現状はウェイトループで対応する
		
		//最終データ送信時に、送信割り込みを「送信エンプティ割り込み」->「送信完了割り込み」に設定
		//(LUOR1.UTIGTS = 1)した場合は、ここでのウェイトは不要となる
		
		//[*重要*]115,200bpsの場合、最大86us程度の待ちが入る事になるので注意
		
		while (MD_ERROR == SCI_SEND(&g_sci_send_buf[g_sci_send_buf_page_index][0], g_sci_send_buf_index[g_sci_send_buf_page_index]) )
		{
			NOP_MCU_DEP;
		}
#else
		SCI_SEND(&g_sci_send_buf[g_sci_send_buf_page_index][0], g_sci_send_buf_index[g_sci_send_buf_page_index]);
#endif
		
		g_sci_send_buf_index[g_sci_send_buf_page_index] = 0;	//格納済みインデックスをクリア
		
		g_sci_send_buf_page_index++;	//送信バッファのページ切り替え
		if (g_sci_send_buf_page_index >= 2) g_sci_send_buf_page_index = 0;
		
		if(i_flag) INTERRUPT_ENABLE;	//割り込み禁止状態で本関数が呼ばれた場合は割り込みを許可しない
		
		NOP_MCU_DEP;
	}
}

void sci_write_uint8_hex(unsigned char c)
{
	//与えられた文字をhex(00-FF)2桁で表示する関数
	
	//引数
	//  unsigned char c : 表示するコード(8bit)
	
	sci_write_char(hex_table_char[((c & 0xf0) >> 4)]);
	sci_write_char(hex_table_char[(c & 0x0f)]);
}

void sci_write_uint16_hex(unsigned short s)
{
	//与えられた文字をhex(0000-FFFF)4桁で表示する関数
	
	//引数
	//  unsigned short s : 表示するコード(16bit)
	
	unsigned char c;
	
	c = (unsigned char)((s & 0xff00) >> 8);
	sci_write_uint8_hex(c);
	
	c = (unsigned char)(s & 0x00ff);
	sci_write_uint8_hex(c);
}

void sci_write_uint32_hex(unsigned long l)
{
	//与えられた文字をhex(00000000-FFFFFFFF)8桁で表示する関数
	
	//引数
	//  unsigned long l : 表示するコード(32bit)
	
	unsigned char c;
	
	c = (unsigned char)((l & 0xff000000) >> 24);
	sci_write_uint8_hex(c);
	
	c = (unsigned char)((l & 0x00ff0000) >> 16);
	sci_write_uint8_hex(c);
	
	c = (unsigned char)((l & 0x0000ff00) >> 8);
	sci_write_uint8_hex(c);
	
	c = (unsigned char)(l & 0x000000ff);
	sci_write_uint8_hex(c);
}

void sci_write_uint8(unsigned char num)
{
	//与えられた数値を10進数で表示する関数
	
	//引数
	//  unsigned char num : 表示させたい数値（符号なし）
	
	unsigned char n[3];
	
	n[0]  = num % 10;
	num  /= 10;
	n[1]  = num % 10;
	num  /= 10;
	n[2]  = num % 10;
	
	if (n[2] != 0)
	{
		sci_write_char(hex_table_char[n[2]]);
		sci_write_char(hex_table_char[n[1]]);
	}
	else if (n[1] != 0)
	{
		sci_write_char(hex_table_char[n[1]]);
	}
	
	sci_write_char(hex_table_char[n[0]]);
}

void sci_write_uint16(unsigned short num)
{
	//与えられた数値を10進数で表示する関数
	
	//引数
	//  unsigned short num : 表示させたい数値（符号なし）
	
	unsigned char n[5];
	unsigned short i, j;
	
	for (i=0; i<5; i++)
	{
		n[i] = (unsigned char)(num % 10);
		if (i == 4) break;
		num /= 10;
	}
	
	for (i=4; i>=1; i--)
	{
		if (n[i] != 0) break;
	}
	
	for (j=i; j>=1; j--) sci_write_char(hex_table_char[n[j]]);
	
	sci_write_char(hex_table_char[n[0]]);
}

void sci_write_uint32(unsigned long num)
{
	//与えられた数値を10進数で表示する関数
	
	//引数
	//  unsigned long num : 表示させたい数値（符号なし）
	
	unsigned char n[10];
	unsigned short i, j;
	
	for (i=0; i<10; i++)
	{
		n[i] = (unsigned char)(num % 10L);
		if (i == 9) break;
		num /= 10L;
	}
	
	for (i=9; i>=1; i--)
	{
		if (n[i] != 0) break;
	}
	
	for (j=i; j>=1; j--) sci_write_char(hex_table_char[n[j]]);
	
	sci_write_char(hex_table_char[n[0]]);
}

void sci_write_int8(char num)
{
	//与えられた数値を10進数で表示する関数
	
	//引数
	//  char num : 表示させたい数値（符号付き）
	
	unsigned char n[3];
	
	if ((num & 0x80) == 0x80)
	{
		sci_write_char('-');
		num = (char)~num;
		num = (char)(num + 1);
	}
	
	n[0]  = (unsigned char)(num % 10);
	num  /= 10;
	n[1]  = (unsigned char)(num % 10);
	num  /= 10;
	n[2]  = (unsigned char)(num % 10);
	
	if (n[2] != 0)
	{
		sci_write_char(hex_table_char[n[2]]);
		sci_write_char(hex_table_char[n[1]]);
	}
	else if (n[1] != 0)
	{
		sci_write_char(hex_table_char[n[1]]);
	}
	
	sci_write_char(hex_table_char[n[0]]);
}

void sci_write_int16(short num)
{
	//与えられた数値を10進数で表示する関数
	
	//引数
	//  short num : 表示させたい数値（符号付き）
	
	if (num < 0)
	{
		sci_write_char('-');
		num = (short)(num * (-1));
	}
	
	sci_write_uint16((unsigned short)num);
}

void sci_write_int32(long num)
{
	//与えられた数値を10進数で表示する関数
	
	//引数
	//  long num : 表示させたい数値（符号付き）
	
	if (num < 0)
	{
		sci_write_char('-');
		num *= -1L;
	}
	
	sci_write_uint32((unsigned long)num);
}

void sci_write_str(const char *str)
{
	//文字列表示関数
	
	//引数
	//  char *str : 表示させたい文字列のポインタ（'\0'終端）
	
	while (*str != '\0')
	{
		if (*str == '\n')    //\nの時は\r\nに変換する
		{
			sci_write_char('\r');
		}
		sci_write_char((unsigned char)*str++);
	}
}

void sci_write_flush(void)
{
	//送信バッファのデータの出力が終わるまで待つ関数
	
	while (g_sci_send_flag != FLAG_CLEAR)
	{
		NOP_MCU_DEP;
	}
}
#endif

#ifdef SCI_USE_RX
unsigned short sci_read_char(unsigned char *c)
{
	//受信バッファ1文字読み出し関数
	
	//引数
	//  unsigned char *c : 受信データ
	//戻り値
	//  0 : 受信データあり
	//  0xFFFF(SCI_RECEIVE_DATA_EMPTY) : 受信データなし
	
	if (g_sci_recv_buf_index1 == g_sci_recv_buf_index2)
	{
		return (unsigned short)SCI_RECEIVE_DATA_EMPTY;
	}

	g_sci_recv_buf_index2++;
	if (g_sci_recv_buf_index2 >= SCI_RECV_BUF_SIZE) g_sci_recv_buf_index2 = 0;
	*c = g_sci_recv_buf[g_sci_recv_buf_index2];
	
	return 0;
}

unsigned short sci_read_str(char *str, unsigned short size)
{
	//受信バッファ文字列読み出し関数
	
	//引数
	//  char *str : 受信データ
	//  unsigned short size : バイト数
	//戻り値
	//  0 : 正常終了
	//  0xFFFF(SCI_RECEIVE_DATA_EMPTY) : 指定されたsizeのデータを受信していない
	
	unsigned short data_receive_size;
	unsigned short i;
	
	data_receive_size = sci_read_data_size();
	
	if (size > data_receive_size)
	{
		return (unsigned short)SCI_RECEIVE_DATA_EMPTY;
	}
	
	for (i=0; i<size; i++)
	{
		g_sci_recv_buf_index2++;
		if (g_sci_recv_buf_index2 >= SCI_RECV_BUF_SIZE) g_sci_recv_buf_index2 = 0;
		
		*str++ = (char)g_sci_recv_buf[g_sci_recv_buf_index2];
	}
	
	return 0;
}

unsigned short sci_read_data_size(void)
{
	//受信バッファ格納済みデータ数出力関数
	
	//戻り値
	//  0 : 受信バッファに受信データなし
	// >0 : 受信バッファに溜まっているデータバイト数
	
	unsigned short ret;
	unsigned short local_sci_recv_buf_index1;
	
	//g_sci_recv_buf_index1は割り込みで随時更新される可能性があるので最初にコピーしておく
	//（途中で更新されても本関数内ではコピーした時点のサイズで処理される）
	local_sci_recv_buf_index1 = g_sci_recv_buf_index1;
	
	if (local_sci_recv_buf_index1 >= g_sci_recv_buf_index2)
	{
		ret = (unsigned short)(local_sci_recv_buf_index1 - g_sci_recv_buf_index2);
	}
	else
	{
		ret = (unsigned short)(SCI_RECV_BUF_SIZE + local_sci_recv_buf_index1 - g_sci_recv_buf_index2);
	}
	
	return ret;
}

void sci_read_buf_clear(void)
{
	//受信バッファに格納されているデータをクリアする関数
	
	g_sci_recv_buf_index2 = g_sci_recv_buf_index1;
}
#endif

#ifdef SCI_USE_FLOAT
int float2str(float value, int num, char *str)
{
	//float文字列変換関数
	
	//戻り値
	//  0 : エラーなし
	// -1 : 引数エラー
	// -2 : 整数部が 2^32-1(4294967295)で表現できる数値を超えている
	
	//引数
	//  float value : 変換元数値
	//  int num     : 小数点以下桁数(0-8)
	//  char *str   : 戻り値ポインタ
	
	float f1;
	float add_f1 = 0.0f;	//四捨五入向けに加算
	unsigned long l1, l2;
	unsigned char buf[10];
	
	int i, j;
	
	*str = '\0';
	
	if (num > 8) return -1;
	if (num < 0) return -1;
	
	//符号
	if (value < 0)
	{
		f1 = value * -1.0f;
		*str++ = '-';
	}
	else
	{
		f1 = value;
	}
	
	//四捨五入処理
	switch (num)
	{
		case 0:
			add_f1 = 0.5f;
			break;
		case 1:
			add_f1 = 0.05f;
			break;
		case 2:
			add_f1 = 0.005f;
			break;
		case 3:
			add_f1 = 0.0005f;
			break;
		case 4:
			add_f1 = 0.00005f;
			break;
		case 5:
			add_f1 = 0.000005f;
			break;
 		case 6:
			add_f1 = 0.0000005f;
			break;
		case 7:
			add_f1 = 0.00000005f;
			break;
		case 8:
			add_f1 = 0.000000005f;
			break;
		default:
			break;
	}
	
	f1 += add_f1;
	
	if (f1 > (float)4294967295UL)
	{
		//オーバフロー
		
		*str++ = 'O';
		*str++ = 'V';
		*str++ = 'F';
		*str = '\0';
		
		return -2;
	}
	
	//整数部
	l1 = (unsigned long)f1;
	
	//   1234567890
	l2 = 1000000000UL;
	
	for (i=0; i<10; i++)
	{
		buf[i] = (unsigned char)(l1 / l2);
		
		if (i == 9) break;
		
		l1 -= buf[i] * l2;
		
		l2 /= 10;
	}
	
	for (i=0; i<10; i++)
	{
		if (buf[i] != 0) break;
	}
	
	if (i == 10)
	{
		*str++ = '0';
	}
	else
	{
		for (j=i; j<10; j++)
		{
			*str++ = (char)(buf[j] + '0');
		}
	}
	
	//小数部
	if (num == 0)
	{
		*str='\0';
		return 0;
	}
	
	switch (num)
	{
		case 1:
			l2 = 10UL;
			break;
		case 2:
			l2 = 100UL;
			break;
		case 3:
			l2 = 1000UL;
			break;
		case 4:
			l2 = 10000UL;
			break;
		case 5:
			l2 = 100000UL;
			break;
		case 6:
			l2 = 1000000UL;
			break;
		case 7:
			l2 = 10000000UL;
			break;
		case 8:
			l2 = 100000000UL;
			break;
		default:
			break;
	}
	
	f1 = f1 - (float)((unsigned long)f1);
	
	l1 = (unsigned long)(f1 * (float)l2);
	l2 /= 10;
	
	for (i=0; i<=(num-1); i++)
	{
		buf[i] = (unsigned char)(l1 / l2);
		
		if (i == (num-1)) break;
		
		l1 -= buf[i] * l2;
		
		l2 /= 10;
	}
	
	*str++ = '.';
	
	for (i=0; i<=(num-1); i++)
	{
		*str++ = (char)(buf[i] + '0');
	}
	
    *str='\0';
	
    return 0;
}

int float2str_eformat(float value, int num, char *str)
{
	//float文字列変換関数, 出力e形式
	
	//戻り値
	//  0 : エラーなし
	// -1 : 引数エラー
	
	//引数
	//  float value : 変換元数値
	//  int num     : 小数点以下桁数(0-8)
	//  char *str   : 戻り値ポインタ
	
	float f1;
	short s = 0;
	unsigned short i;
	char buf[20];
	int ret;
	
	*str = '\0';
	
	if (num > 8) return -1;
	if (num < 0) return -1;
	
	//符号
	if (value < 0)
	{
		f1 = value * -1.0f;
		*str++ = '-';
	}
	else
	{
		f1 = value;
	}
	
	if (f1 > 10.0f)
	{
		for (i=0; i<39; i++)
		{
			f1 /= 10.0f;
			s++;
			if (f1 < 10.0f) break;
		}
	}
	else if (f1 < 1.0f)
	{
		for (i=0; i<39; i++)
		{
			f1 *= 10.0f;
			s--;
			if (f1 > 1.0f) break;
		}
	}
	
	ret = float2str(f1, num, buf);
	if (ret != 0)
	{
		*str = '\0';
		return ret;
	}
	
	i = 0;
	while (buf[i] != '\0')
	{
		*str++ = buf[i++];
	}
	
	*str++ = 'e';
	
	ret = float2str((float)s, 0, buf);
	if (ret != 0)
	{
		*str = '\0';
		return ret;
	}
	
	i = 0;
	while (buf[i] != '\0')
	{
		*str++ = buf[i++];
	}
		
	*str='\0';
	
	return 0;
}
	
int double2str(double value, int num, char *str)
{
	//double文字列変換関数
	
	//戻り値
	//  0 : エラーなし
	// -1 : 引数エラー
	// -2 : 整数部が 2^32-1(4294967295)で表現できる数値を超えている
	
	//引数
	//  double value : 変換元数値
	//  int num     : 小数点以下桁数(0-8)
	//  char *str   : 戻り値ポインタ
	
	double d1;
	double add_d1 = 0.0;	//四捨五入向けに加算
	unsigned long l1, l2;
	unsigned char buf[10];
	
	int i, j;
	
	*str = '\0';
	
	if (num > 8) return -1;
	if (num < 0) return -1;
	
	//符号
	if (value < 0)
	{
		d1 = value * -1.0;
		*str++ = '-';
	}
	else
	{
		d1 = value;
	}
	
	//四捨五入処理
	switch (num)
	{
		case 0:
			add_d1 = 0.5;
			break;
		case 1:
			add_d1 = 0.05;
			break;
		case 2:
			add_d1 = 0.005;
			break;
		case 3:
			add_d1 = 0.0005;
			break;
		case 4:
			add_d1 = 0.00005;
			break;
		case 5:
			add_d1 = 0.000005;
			break;
 		case 6:
			add_d1 = 0.0000005;
			break;
		case 7:
			add_d1 = 0.00000005;
			break;
		case 8:
			add_d1 = 0.000000005;
			break;
		default:
			break;
	}
	
	d1 += add_d1;
	
	if (d1 > 4294967295UL)
	{
		//オーバフロー
		
		*str++ = 'O';
		*str++ = 'V';
		*str++ = 'F';
		*str = '\0';
		
		return -2;
	}
	
	//整数部
	l1 = (unsigned long)d1;
	
	//   1234567890
	l2 = 1000000000UL;
	
	for (i=0; i<10; i++)
	{
		buf[i] = (unsigned char)(l1 / l2);
		
		if (i == 9) break;
		
		l1 -= buf[i] * l2;
		
		l2 /= 10;
	}
	
	for (i=0; i<10; i++)
	{
		if (buf[i] != 0) break;
	}
	
	if (i == 10)
	{
		*str++ = '0';
	}
	else
	{
		for (j=i; j<10; j++)
		{
			*str++ = (char)(buf[j] + '0');
		}
	}
	
	//小数部
	if (num == 0)
	{
		*str='\0';
		return 0;
	}
	
	switch (num)
	{
		case 1:
			l2 = 10UL;
			break;
		case 2:
			l2 = 100UL;
			break;
		case 3:
			l2 = 1000UL;
			break;
		case 4:
			l2 = 10000UL;
			break;
		case 5:
			l2 = 100000UL;
			break;
		case 6:
			l2 = 1000000UL;
			break;
		case 7:
			l2 = 10000000UL;
			break;
		case 8:
			l2 = 100000000UL;
			break;
		default:
			break;
	}
	
	d1 = d1 - (double)((unsigned long)d1);
	
	l1 = (unsigned long)(d1 * (double)l2);
	l2 /= 10;
	
	for (i=0; i<=(num-1); i++)
	{
		buf[i] = (unsigned char)(l1 / l2);
		
		if (i == (num-1)) break;
		
		l1 -= buf[i] * l2;
		
		l2 /= 10;
	}
	
	*str++ = '.';
	
	for (i=0; i<=(num-1); i++)
	{
		*str++ = (char)(buf[i] + '0');
	}
	
    *str='\0';
	
    return 0;
}

int double2str_eformat(double value, int num, char *str)
{
	//double文字列変換関数, 出力e形式
	
	//戻り値
	//  0 : エラーなし
	// -1 : 引数エラー
	
	//引数
	//  double value : 変換元数値
	//  int num     : 小数点以下桁数(0-8)
	//  char *str   : 戻り値ポインタ
	
	double d1;
	short s = 0;
	unsigned short i;
	char buf[20];
	int ret;
	
	*str = '\0';
	
	if (num > 8) return -1;
	if (num < 0) return -1;
	
	//符号
	if (value < 0)
	{
		d1 = value * -1.0;
		*str++ = '-';
	}
	else
	{
		d1 = value;
	}
	
	if (d1 > 10.0)
	{
		for (i=0; i<309; i++)
		{
			d1 /= 10.0;
			s++;
			if (d1 < 10.0) break;
		}
	}
	else if (d1 < 1.0)
	{
		for (i=0; i<309; i++)
		{
			d1 *= 10.0f;
			s--;
			if (d1 > 1.0f) break;
		}
	}
	
	ret = double2str(d1, num, buf);
	if (ret != 0)
	{
		*str = '\0';
		return ret;
	}
	
	i = 0;
	while (buf[i] != '\0')
	{
		*str++ = buf[i++];
	}
	
	*str++ = 'e';
	
	ret = double2str((double)s, 0, buf);
	if (ret != 0)
	{
		*str = '\0';
		return ret;
	}
	
	i = 0;
	while (buf[i] != '\0')
	{
		*str++ = buf[i++];
	}
		
	*str='\0';
	
	return 0;
}
#endif

/*----------------------------------------------------------------------
	ソースコード（ローカル）
----------------------------------------------------------------------*/

static int interrupt_flag_get(void)
{
	int ret;
	
#if defined(MCU_TYPE_RX)

	unsigned long psw;
	
	psw = __get_psw() & 0x00010000UL;
	
	ret = psw >> 16UL;
	
#endif

#if defined(MCU_TYPE_RL78)

	unsigned char psw;
	
	psw = __get_psw();
	
	ret = psw >> 7;
	
#endif

#if defined(MCU_TYPE_RA)
	
	ret = __get_PRIMASK() ? 0 : 1;
	
#endif

#if defined(MCU_TYPE_RH850)

	unsigned long psw;
	
	psw = __stsr(5) & 0x00000020UL;
	
	ret = psw ? 0 : 1;
	
#endif

#if defined(MCU_TYPE_RISC_V)

	ret = 1;	//CPU仕様で割り込みは常に許可設定ではないが、ここでは「全体の割り込み許可区分として」常に割り込み許可として処理

#endif

	return ret;
}

/*----------------------------------------------------------------------
	ソースコード（割り込みで呼び出される関数）
----------------------------------------------------------------------*/
#ifdef SCI_USE_TX
void intr_sci_send_end(void)
{
	//送信完了時に実行する処理
	
	int i_flag = interrupt_flag_get();
	
	if(i_flag) INTERRUPT_DISABLE;	//割込みフラグにより処理内容を変える必要性はないが、元々割込み禁止の時は何もしない
	
	//送信完了時にウラ（データ格納用）バッファにデータがない
	if (g_sci_send_buf_index[g_sci_send_buf_page_index] == 0)
	{
		g_sci_send_flag = FLAG_CLEAR;
	}
	else
	{
		//送信関数をキックする
		
#if defined(MCU_TYPE_RH850)
		
		//前述「-RH850での送信ライブラリ関数呼び出し時の注意点-」参照
		
		while(MD_ERROR == SCI_SEND(&g_sci_send_buf[g_sci_send_buf_page_index][0], g_sci_send_buf_index[g_sci_send_buf_page_index]) )
		{
			NOP_MCU_DEP;
		}
#else
		SCI_SEND(&g_sci_send_buf[g_sci_send_buf_page_index][0], g_sci_send_buf_index[g_sci_send_buf_page_index]);
#endif
		
		g_sci_send_buf_index[g_sci_send_buf_page_index] = 0;	//格納済みインデックスをクリア
		
		g_sci_send_buf_page_index++;	//送信バッファのページ切り替え
		if (g_sci_send_buf_page_index >= 2) g_sci_send_buf_page_index = 0;
	}
	
	if(i_flag) INTERRUPT_ENABLE;	//割り込み禁止状態で本関数が呼ばれた場合は割り込みを許可しない
}
#endif

#ifdef SCI_USE_RX
void intr_sci_receive_end(void)
{
	//受信完了時に実行する処理
	
	g_sci_recv_buf_index1++;
	if (g_sci_recv_buf_index1 >= SCI_RECV_BUF_SIZE) g_sci_recv_buf_index1 = 0;

	g_sci_recv_buf[g_sci_recv_buf_index1] = g_sci_recv_data;

	if (g_sci_recv_buf_index1 == g_sci_recv_buf_index2)      //書き込みインデックスが読み出しインデックスを超えた
	{
		g_sci_recv_buf_index2 = (unsigned short)(g_sci_recv_buf_index1 + 1);
		if (g_sci_recv_buf_index2 >= SCI_RECV_BUF_SIZE) g_sci_recv_buf_index2 = 0;
		//この時点で未読み出しのデータは捨てられるが、最新の SCI_RECV_BUF_SIZE のデータは未読み出しで有効とする
		g_sci_status |= SCI_RECEIVE_BUF_OVERFLOW;
	}
	
	//次のデータを受信する
    SCI_RECEIVE(&g_sci_recv_data, 1);
}

void intr_sci_receive_error(void)
{
	//受信エラー時に実行する処理
	
	//スマートコンフィグレータ/FSP生成コード内でSSRのエラークリアは行っている
	
	//RL78の場合は、SSRのエラークリア後に本関数が呼ばれる
	//※RL78はソフトウェアオーバランの場合コールバック関数が呼ばれるが本プログラムでは未使用
	
	//RXの場合は、SSRエラークリア前に本関数が呼ばれる
	//※本関数でSSRを確認すれば、エラー内容は判るがSCI-chに応じて見るレジスタを変えなければならないので
	//本プログラムではエラー種別の取得は行っていない
	
	//RAの場合は、SSRのエラークリア後に本関数が呼ばれるがエラーフラグは保持されている
	
	g_sci_status |= SCI_RECEIVE_ERROR;
	
}
#endif

#if defined(MCU_TYPE_RA)

//RAのUART割り込み処理

void user_uart_callback (uart_callback_args_t * p_args)
{
	/* Handle the UART event */
	switch (p_args->event)
	{
		/* Received a character */
		case UART_EVENT_RX_CHAR:
		{
			//9bit以上のデータを受信する際は処理を記載
			break;
		}
		/* Receive complete */
		case UART_EVENT_RX_COMPLETE:
		{
#ifdef SCI_USE_RX
			intr_sci_receive_end();
#endif
			break;
		}
		/* Transmit complete */
		case UART_EVENT_TX_COMPLETE:
		{
#ifdef SCI_USE_TX
			intr_sci_send_end();
#endif
			break;
		}
		/* error */
		case UART_EVENT_ERR_FRAMING:
		{
#ifdef SCI_USE_RX
			g_sci_status |= SCI_RECEIVE_FRAMING_ERROR;
			intr_sci_receive_error();
#endif
			break;
		}
		case UART_EVENT_ERR_OVERFLOW:
		{
#ifdef SCI_USE_RX
			g_sci_status |= SCI_RECEIVE_OVERRUN_ERROR;
			intr_sci_receive_error();
#endif
			break;
		}
		default:
		{
		}
	}
}
#endif
