/*
 * @file  intr.c
 * @brief HSBRX65_IO_BOARD割り込み処理
 *
 * 本ソフトウェアの著作権は作成元である(株)北斗電子が所有するものとし、
 * (株)北斗電子は、以下の (1)-(3) の条件を満たす場合に限り、
 * 本ソフトウェア（本ソフトウェアを改変したものを含む。以下同じ）を
 * 使用・複製・改変・再配布（以下、利用と呼ぶ）することを無償で許諾する。
 *
 * (1) 本ソフトウェアをソースコードの形で利用する場合には、下記の著作
 *	   権表示、この利用条件が、そのままの形でソースコード中に含まれて
 *	   いること。
 * (2) 本ソフトウェアの一部または全てを無断で転載することを禁止するもの
 *	   とする。雑誌などへ紹介・収録の場合は(株)北斗電子に連絡願います。
 * (3) 本ソフトウェアの利用により直接的または間接的に生じるいかなる損害
 *	   からも、(株)北斗電子は一切の責任を負わないものとする。
 *
 * Copyright (C) Hokuto denshi Co,Ltd. 2024
 */

/*----------------------------------------------------------------------
	インクルード
----------------------------------------------------------------------*/
#include	"r_smc_entry.h"
#include	"../io_board/io_board.h"

/*----------------------------------------------------------------------
	割り込みコールバック関数プロトタイプ宣言
----------------------------------------------------------------------*/
void (*cmt3_callback) (void);
void (*rtc_callback) (void);
void (*adc_callback) (void);
void (*sw8_callback) (void);
void (*sw9_callback) (void);
void (*sw10_callback) (void);
void (*tpu2a_callback) (void);
void (*tpu2b_callback) (void);

/*----------------------------------------------------------------------
	割り込み関数
----------------------------------------------------------------------*/
void intr_cmt3(void)
{
	if (g_function_use[IOBOARD_ADC] == ENABLE)
	{
		//A/D変換の起動を行う
	
		if (g_adc_flag == 0)			//A/D変換中でない場合
		{
			R_Config_S12AD0_Start();	//A/D変換を起動
			g_adc_flag = 1;				//A/D変換中フラグを立てる
		}
	}
	
	if (g_function_use[IOBOARD_LED7SEG] == ENABLE)
	{
		//7セグLEDの表示桁の移動を行う
		
		/*
		 *	PA0-PA7 セグ信号
		 *  P74-P77 COM信号
		 *  P70-P73のPODR値は変更しない
		 */
		
		static int com = 0;
		
		switch (com)
		{
			case 1:
			
				//COM1のSEGデータセット & P74(COM1)を選択
				PORTA.PODR.BYTE = 0x00;									//一旦消灯
				PORT7.PODR.BYTE &= 0x0F;
				PORT7.PODR.BYTE |= 0x10;								//COM1選択
				PORTA.PODR.BYTE = g_led7seg_data[IOBOARD_LED7SEG_COM1];	//COM1のデータをセグ信号にセット
				com = 2;												//次に表示するCOM
				break;
				
			case 2:
			
				//COM2のSEGデータセット & P75(COM2)を選択
				PORTA.PODR.BYTE = 0x00;									//一旦消灯
				PORT7.PODR.BYTE &= 0x0F;
				PORT7.PODR.BYTE |= 0x20;								//COM2選択
				PORTA.PODR.BYTE = g_led7seg_data[IOBOARD_LED7SEG_COM2];	//COM2のデータをセグ信号にセット
				com = 3;												//次に表示するCOM
				break;
				
			case 3:
			
				//COM3のSEGデータセット & P76(COM3)を選択
				PORTA.PODR.BYTE = 0x00;									//一旦消灯
				PORT7.PODR.BYTE &= 0x0F;
				PORT7.PODR.BYTE |= 0x40;								//COM3選択
				PORTA.PODR.BYTE = g_led7seg_data[IOBOARD_LED7SEG_COM3];	//COM3のデータをセグ信号にセット
				com = 4;												//次に表示するCOM
				break;

			case 4:
			
				//COM4のSEGデータセット & P77(COM4)を選択
				PORTA.PODR.BYTE = 0x00;									//一旦消灯
				PORT7.PODR.BYTE &= 0x0F;
				PORT7.PODR.BYTE |= 0x80;								//COM4選択
				PORTA.PODR.BYTE = g_led7seg_data[IOBOARD_LED7SEG_COM4];	//COM4のデータをセグ信号にセット
				com = 1;												//次に表示するCOM
				break;
			
			default:	//開始時COM1を選択
				com = 1;
				break;
		}
	}
	
	if (g_function_use[IOBOARD_MATRIXSW] == ENABLE)
	{
		//マトリックススイッチの読み取りとROW選択の変更を行う
		
		/*
		 *	P60-P63 COL信号（入力）
		 *  P64-P67 ROW信号（出力）
		 */ 
		
		static int row = 0;
	
		unsigned char port6 = PORT6.PIDR.BYTE;
		
		switch (row)
		{
			case 1:
	
				//P64(ROW1)が選択されている状態→SW11-SW14の読み取り & P65(ROW2)を選択
				g_matrixsw[IOBOARD_SW11] = port6 & 0x1;
				g_matrixsw[IOBOARD_SW12] = (port6 & 0x2) ? 1 : 0;
				g_matrixsw[IOBOARD_SW13] = (port6 & 0x4) ? 1 : 0;
				g_matrixsw[IOBOARD_SW14] = (port6 & 0x8) ? 1 : 0;

				PORT6.PODR.BYTE = 0xD0;			//ROW2を選択
				row = 2;						//ROW2選択の情報を保持
				break;
				
			case 2:

				//P65(ROW2)が選択されている状態→SW15-SW18の読み取り & P66(ROW3)を選択
				g_matrixsw[IOBOARD_SW15] = port6 & 0x1;
				g_matrixsw[IOBOARD_SW16] = (port6 & 0x2) ? 1 : 0;
				g_matrixsw[IOBOARD_SW17] = (port6 & 0x4) ? 1 : 0;
				g_matrixsw[IOBOARD_SW18] = (port6 & 0x8) ? 1 : 0;
		
				PORT6.PODR.BYTE = 0xB0;			//ROW3を選択
				row = 3;						//ROW3選択の情報を保持
				break;

			case 3:
			
				//P66(ROW3)が選択されている状態→SW19-SW22の読み取り & P67(ROW4)を選択
				g_matrixsw[IOBOARD_SW19] = port6 & 0x1;
				g_matrixsw[IOBOARD_SW20] = (port6 & 0x2) ? 1 : 0;
				g_matrixsw[IOBOARD_SW21] = (port6 & 0x4) ? 1 : 0;
				g_matrixsw[IOBOARD_SW22] = (port6 & 0x8) ? 1 : 0;
		
				PORT6.PODR.BYTE = 0x70;			//ROW4を選択
				row = 4;						//ROW4選択の情報を保持
				break;

			case 4:
				//P67(ROW4)が選択されている状態→SW23-SW26の読み取り & P64(ROW1)を選択
				g_matrixsw[IOBOARD_SW23] = port6 & 0x1;
				g_matrixsw[IOBOARD_SW24] = (port6 & 0x2) ? 1 : 0;
				g_matrixsw[IOBOARD_SW25] = (port6 & 0x4) ? 1 : 0;
				g_matrixsw[IOBOARD_SW26] = (port6 & 0x8) ? 1 : 0;
		
				PORT6.PODR.BYTE = 0xE0;			//ROW1を選択
				row = 1;						//ROW1選択の情報を保持
				break;
				
			default:
				PORT6.PODR.BYTE = 0xE0;			//開始時P64(ROW1)を選択
				row = 1;
				break;
		}
	}
	
	//コールバック関数の中身はアプリケーション毎に定義する
	cmt3_callback();
}

void intr_rtc(void)
{
	setpsw_i();		//多重割り込み許可
	
	//コールバック関数の中身はアプリケーション毎に定義する
	rtc_callback();
}

void intr_adc(void)
{
	//A/D変換終了割り込み関数
	
	unsigned short result;
	static unsigned short index = 0;
	static unsigned short data_history_buf[IOBOARD_ADC_AVERAGE_NUM] = {0};
	static unsigned long sum = 0;
	
	R_Config_S12AD0_Get_ValueResult(ADCHANNEL0, &result);	//A/D変換結果の読み出し
	
	g_adc_flag = 0;						//A/D変換中フラグを落とす
	
	setpsw_i();		//多重割り込み許可
	
#if (IOBOARD_ADC_AVERAGE_NUM > 1)

	//移動平均の算出
	
	//一番古いデータを合計から省く
	sum -= data_history_buf[index];
	
	//履歴バッファに現在のデータを格納
	data_history_buf[index] = result;
	
	//移動平均の合計値に最新のデータを加算
	sum += data_history_buf[index];
	
	//インデックスを進める
	index++;
	if (index >= IOBOARD_ADC_AVERAGE_NUM)
	{
		index = 0;
	}
	
	/*
	 * data_history_bufへのデータ格納が一巡するまでは、初期値0からの移動平均となる
	 */
	 
	g_adc_result = (unsigned short)(sum / IOBOARD_ADC_AVERAGE_NUM);		//A/D変換結果を移動平均値とする
	
	/*
	 * ADC_AVERAGE_NUM は2のべき乗となる値が望ましい（最適化により除算がビットシフト命令に置き換えられる）
	 */

#else

	//移動平均を使用しない
	
	g_adc_result = result;

#endif

	//コールバック関数の中身はアプリケーション毎に定義する
	adc_callback();
}

void intr_irq4(void)
{
	//SW10割り込み関数
	
	setpsw_i();		//多重割り込み許可
	
	//コールバック関数の中身はアプリケーション毎に定義する
	sw10_callback();
}

void intr_irq13(void)
{
	//SW9割り込み関数
	
	setpsw_i();		//多重割り込み許可
	
	//コールバック関数の中身はアプリケーション毎に定義する
	sw9_callback();
}

void intr_irq15(void)
{
	//SW8割り込み関数
	
	setpsw_i();		//多重割り込み許可
	
	//コールバック関数の中身はアプリケーション毎に定義する
	sw8_callback();
}

void intr_tpu2a(void)
{
	//TPU2の周期終わり
	
	//TGRA/Bレジスタの更新
	
	/*
	 * TGRAレジスタは随時タイミングではなく、周期終わりTGRAコンペアマッチが終わったタイミングで更新
	 */
	 
	if (TPU2.TGRA != g_tpu2_tgra)
	{
		TPU2.TGRA = g_tpu2_tgra;
	}
	
	/*
	 * TGRBレジスタは、タイマカウント値が1000のタイミングで
	 * 2000→500の様に変更（dutyを小さくする変更）した場合は、周期終わりまでコンペアマッチが発生しなくなる
	 * （dutyを小さい方向に変更したにも拘わらず、1周期以上長い出力タイミングが生じてしまう）
	 * 値変更の1周期のタイミングで、周期の伸びが生じるケースがあるので、レジスタ値の変更は決まったタイミングで行う
	 */
	
	if (TPU2.TGRB != g_tpu2_tgrb)
	{
		TPU2.TGRB = g_tpu2_tgrb;
	}
	
	//コールバック関数の中身はアプリケーション毎に定義する
	tpu2a_callback();
}

void intr_tpu2b(void)
{
	//TPU2のコンペアマッチB
	
	//コールバック関数の中身はアプリケーション毎に定義する
	tpu2b_callback();
}
