/*
 * @file  io_board.c
 * @brief HSBRX65_IO_BOARDドライバ
 *
 * 本ソフトウェアの著作権は作成元である(株)北斗電子が所有するものとし、
 * (株)北斗電子は、以下の (1)-(3) の条件を満たす場合に限り、
 * 本ソフトウェア（本ソフトウェアを改変したものを含む。以下同じ）を
 * 使用・複製・改変・再配布（以下、利用と呼ぶ）することを無償で許諾する。
 *
 * (1) 本ソフトウェアをソースコードの形で利用する場合には、下記の著作
 *	   権表示、この利用条件が、そのままの形でソースコード中に含まれて
 *	   いること。
 * (2) 本ソフトウェアの一部または全てを無断で転載することを禁止するもの
 *	   とする。雑誌などへ紹介・収録の場合は(株)北斗電子に連絡願います。
 * (3) 本ソフトウェアの利用により直接的または間接的に生じるいかなる損害
 *	   からも、(株)北斗電子は一切の責任を負わないものとする。
 *
 * Copyright (C) Hokuto denshi Co,Ltd. 2024
 */

/*----------------------------------------------------------------------
	インクルード
----------------------------------------------------------------------*/
#include	"r_smc_entry.h"
#include	"io_board.h"

/*----------------------------------------------------------------------
	グローバル変数
----------------------------------------------------------------------*/
//CMT3で処理する各機能を有効化するか否か
volatile int g_function_use[IOBOARD_CMT3_FUNCTION_NUM] = {0};

//ADC向け
volatile int g_adc_flag = 0;
volatile unsigned short g_adc_result;

//7セグLED向け
volatile unsigned char g_led7seg_data[IOBOARD_LED7SEG_COM] = {0};

//マトリックススイッチ
volatile unsigned char g_matrixsw[IOBOARD_MATRIXSW_NUM] = {0};

//TPU2.TGRA/Bレジスタ
volatile unsigned short g_tpu2_tgra = 60000 - 1;	//周期1.024秒
volatile unsigned short g_tpu2_tgrb = 30000 - 1;	//duty 50&

/*----------------------------------------------------------------------
	関数プロトタイプ宣言
----------------------------------------------------------------------*/
//プッシュスイッチ(SW0-SW7)
int push_sw_init(void);
unsigned char push_sw_read(void);

//プッシュスイッチ(SW8-SW10), 割り込みで処理されるスイッチ
int intr_push_sw_init(int timing);
int intr_push_sw_start(void);
int intr_push_sw_stop(void);

//LED(LED0-LED7)
int led_init(void);
int led_set(unsigned char led);

//ブザー(B1), JP2:P17（上側ショート）
int buzzer_freq_set(unsigned short freq);
int buzzer_on(void);
int buzzer_off(void);

//LED8, JP1:P15（下側ショート）
int led8_cycle_set(unsigned short cycle);
int led8_duty_set(float duty);
int led8_on(void);
int led8_off(void);

//ステッピングモータ
int stepping_motor_init(void);
int stepping_motor_start(int direction);
int stepping_motor_stop(void);
int stepping_motor_step_move(int direction);
int stepping_motor_drive(unsigned char pattern);

//ADC
int adc_start(void);
int adc_stop(void);
unsigned short adc_val(void);

//7セグLED
int led7seg_init(void);
int led7seg_start(void);
int led7seg_stop(void);
int led7seg_segument_data_set(unsigned char column, unsigned char seg, unsigned char seg_on_off);
int led7seg_disp_hex(unsigned short data);
int led7seg_disp_num(short data, unsigned char dp);

//マトリックススイッチ
int matrixsw_init(void);
int matrixsw_start(void);
int matrixsw_stop(void);

/*----------------------------------------------------------------------
	関数定義
----------------------------------------------------------------------*/
int push_sw_init(void)
{
	//プッシュスイッチ初期化関数
	
	//SW0-SW7の初期化を行います
	
	//引数
	//  なし
	
	//戻り値
	//  RET_SUCCESS(0)
	
	PORTD.PDR.BYTE = 0x00;	//端子を入力モードにセット
	
	return RET_SUCCESS;
}

unsigned char push_sw_read(void)
{
	//プッシュスイッチ読み取り関数
	
	//SW0-SW7の状態を読み取ります
	
	//引数
	//  なし
	
	//戻り値
	//  SW読み取り結果(OFF:bit=1, ON:bit=0)
	
	return PORTD.PIDR.BYTE;
}

int intr_push_sw_init(int timing)
{
	//割り込み対応プッシュスイッチ初期化関数
	
	//SW8-SW10の割り込みのエッジを設定します
	
	//引数
	//  timing
	//    0 : ボタンを押した際に割り込みが掛ります
	//    1 : ボタンを押し込んでいる状態から離した際に割り込みが掛ります
	//    2 : 押した際、離した際の両方割り込みが掛ります
	
	//戻り値
	//  正常終了 : RET_SUCCESS(0)
	//  引数エラー : RET_ERROR(-1)
	
	/*
	 * 本関数を実行しない場合スマート・コンフィグレータ設定の値（押した際に割り込み）となります
	 */
	 
	 unsigned char val;
	 
	 if (timing == 0)
	 {
		 val = 0x1;		//立下りエッジ
	 }
	 else if (timing == 1)
	 {
		 val = 0x2;		//立ち上がりエッジ
	 }
	 else if (timing == 2)
	 {
		 val = 0x3;		//両エッジ
	 }
	 else
	 {
		 return RET_ERROR;
	 }
	 
	ICU.IRQCR[4].BIT.IRQMD = val;
	ICU.IRQCR[13].BIT.IRQMD = val;
	ICU.IRQCR[15].BIT.IRQMD = val;
	
	return RET_SUCCESS;
	
}

int intr_push_sw_start(void)
{
	//割り込み対応プッシュスイッチ動作開始関数
	
	//SW8-SW10の割り込みの有効化を行います
	
	//引数 なし
	
	//戻り値
	//  RET_SUCCESS(0)
	
	R_Config_ICU_IRQ4_Start();		//SW10
	R_Config_ICU_IRQ13_Start();		//SW9
	R_Config_ICU_IRQ15_Start();		//SW8
	
	return RET_SUCCESS;
}

int intr_push_sw_stop(void)
{
	//割り込み対応プッシュスイッチ動作停止関数
	
	//SW8-SW10の割り込みの無効化を行います
	
	//引数 なし
	
	//戻り値
	//  RET_SUCCESS(0)
	
	R_Config_ICU_IRQ4_Stop();		//SW10
	R_Config_ICU_IRQ13_Stop();		//SW9
	R_Config_ICU_IRQ15_Stop();		//SW8
	
	return RET_SUCCESS;
}

int led_init(void)
{
	//LED初期化関数
	
	//LED0-LED7の初期化を行います
	
	//引数
	//  なし
	
	//戻り値
	//  RET_SUCCESS(0)
	
	PORTE.PODR.BYTE = 0xFF;		//初期状態は消灯
	PORTE.PDR.BYTE = 0xFF;
	
	return RET_SUCCESS;
}

int led_set(unsigned char led)
{
	//LED設定関数
	
	//LED0-LED7の状態を設定します
	
	//引数
	//  led : LED設定値(bit=0:OFF, bit=1:ON)
	
	//戻り値
	//  RET_SUCCESS(0)
	
	PORTE.PODR.BYTE = ~led;
	
	return RET_SUCCESS;
}

int buzzer_freq_set(unsigned short freq)
{
	//ブザーの周波数設定関数
	
	//引数
	//  freq : 設定周波数
	
	//戻り値
	//  RET_SUCCESS(0)
	
	/*
	 * 設定可能な周波数は300-8000[Hz]ですが
	 * 8bitタイマを使用しているため、設定刻みは荒くなります
	 * 300未満の値を設定した場合は、300に
	 * 8000以上の値を設定した場合は、8000に設定します
	 * （TMRタイマとしては、228.9-29.3kHzの範囲）
	 */
	 
	 const unsigned short freq_min = 300;
	 const unsigned short freq_max = 8000;
	 
	 unsigned char reg;
	 
	 if (freq < freq_min) freq = freq_min;
	 else if (freq > freq_max) freq = freq_max;
	 
	 float clock_cycle_time = 1.0f / (PCLKB * 1.0e6f) * 1024.0f;	//TMRタイマの1カウントのサイクル時間, 1024分周
	 float cycle_time = 1.0f / freq;
	 
	 reg = (unsigned char)(cycle_time / clock_cycle_time) - 1;
	 
	 TMR1.TCORB = reg / 2;		//コンペアマッチBレジスタは周期レジスタの1/2に設定する（常にduty 50%で駆動）
	 TMR1.TCORA = reg;
	 
	 /*
	  * TCORB > TCORAとなると、波形はトグルしない
	  * TCORBを先に更新する
	  */
	 
	 /*
	  * TCORA,TCORBの書き換えタイミングでは、コンペアマッチは行われないため
	  * 本関数が呼ばれたタイミングで波形に抜けが生じるケースがある
	  * 本関数が非常に短い周期で連続で呼ばれた場合、出力波形に明らかに抜けが生じる
	  * →その場合は、ブザー音の異常が認識できるレベル
	  * （波形抜けを抑止したい場合は、レジスタの更新タイミングを随時→コンペアマッチ割り込みの後
	  *  に変えるといった対策があるが、本関数では関数呼び出し時の一瞬の波形抜けは許容する）
	  */
	 
	return RET_SUCCESS;
}

int buzzer_on(void)
{
	//ブザー鳴動開始
	
	//引数
	//  なし
	
	//戻り値
	//  RET_SUCCESS(0)
	
	R_Config_TMR1_Start();
	
	return RET_SUCCESS;
}

int buzzer_off(void)
{
	//ブザー鳴動停止
	
	//引数
	// なし
	
	//戻り値
	//  RET_SUCCESS(0)
	
	R_Config_TMR1_Stop();
	
	return RET_SUCCESS;
}

int led8_cycle_set(unsigned short cycle)
{
	//LED8の周期設定関数
	
	//引数
	//  cycle : サイクル数(1-65535)
	
	//戻り値
	//  RET_SUCCESS(0)
	
	/*
	 * TGRBレジスタは、タイマカウント値が1000のタイミングで
	 * 2000→500の様に変更した場合は、周期終わりまでコンペアマッチが発生しなくなる
	 * 値変更の1周期のみ、周期の伸びが生じるケースがあるので、レジスタ値の変更は決まったタイミングで行う
	 */
	 
	/*
	 * ※本関数内のdutyは正の（H期間）を基準とした値
	 */
	 
	if (cycle == 0) cycle = 1;
	
	/*
	 * 周期(TGRA)の最低は1
	 * コンペアマッチ(TGRB)の最低は0
	 * → TGRA=1, TGRB=0の時は、周期2サイクル, 切り換えタイミング1サイクル（29kHz, duty=0.5）
	 * dutyの計算は、TGRxに1を加算した値で行う
	 */
	
	float duty = (float)(g_tpu2_tgrb + 1) / (float)(g_tpu2_tgra + 1);
	
	g_tpu2_tgra = cycle;
	g_tpu2_tgrb = (unsigned short)(cycle * duty);		//周期を変更してもdutyは変更前の値を維持

	return RET_SUCCESS;
}

int led8_duty_set(float duty)
{
	//LED8の周期設定関数
	
	//引数
	//  duty : duty比(0-1)
	
	//戻り値
	//  RET_SUCCESS(0)
	
	/*
	 * 負の値を指定した場合は0
	 * 1以上の値を指定した場合は、1に設定します
	 */
	 
	 if (duty < 0) duty = 0.0f;
	 else if (duty > 1) duty = 1.0f;
	 
	 /*
	  * 周期始まりでH, コンペアマッチでL出力なので、
	  * TGRB = TGRA（周期） * duty
	  * となるが、LED8はLで点灯なので
	  * 1.0 - dutyでレジスタ値を決める
	  * duty <-> TGRB : 反転
	  */
	  
	 g_tpu2_tgrb = (unsigned short)(g_tpu2_tgra * (1.0f - duty));
	 
	 return RET_SUCCESS;
}

int led8_on(void)
{
	//LED8の点滅開始
	
	//引数
	//  なし
	
	//戻り値
	//  RET_SUCCESS(0)
	
	R_Config_TPU2_Start();
	
	return RET_SUCCESS;
}

int led8_off(void)
{
	//LED8の点滅停止
	
	//引数
	//  なし
	
	//戻り値
	//  RET_SUCCESS(0)
	
	unsigned char tior;
	
	R_Config_TPU2_Stop();
	
	//LED8が点灯中にタイマが停止した場合、LED8を消灯させる
	
	tior = TPU2.TIOR.BYTE;
	
	TPU2.TIOR.BYTE = 0;
	
	TPU2.TIOR.BYTE = tior;

	return RET_SUCCESS;
}

int stepping_motor_init(void)
{
	//ステッピングモータ初期化
	
	//引数
	//  なし
	
	//戻り値
	//  RET_SUCCESS(0)
	
	/*
	 * PB4-PB7:出力端子
	 */
	 
	PORTB.PODR.BYTE &= 0x0F;	//PB4-PB7出力=L
	PORTB.PDR.BYTE |= 0xF0;		//PB4-PB7出力有効
	
	return RET_SUCCESS;
}

int stepping_motor_stop(void)
{
	//ステッピングモータ回転停止
	
	//ステッピングモータ駆動トランジスタをOFF制御します
	
	//引数
	//  なし
	
	//戻り値
	//  RET_SUCCESS(0)

	
	PORTB.PODR.BYTE &= 0x0F;			//PB4-PB7出力=L
	
	return RET_SUCCESS;
}

int stepping_motor_step_move(int direction)
{
	//ステッピングモータのステップを変更します
	
	//引数
	//  direction
	//    1 : 正回転
	//    -1 : 逆回転
	
	//戻り値
	//  RET_SUCCESS(0)
	//  引数エラー : RET_ERROR(-1)
	
	static int local_pattern = 0;
	
	if ((direction != 1) && (direction != -1))
	{
		return RET_ERROR;
	}
	
	//パターンをdirectionに応じて変える
	if (direction == 1)
	{
		local_pattern++;
	}
	else if (direction == -1)
	{
		local_pattern--;
	}
	
	//パターンは0-3の範囲とする
	if (local_pattern >= 4)
	{
		local_pattern = 0;
	}
	else if (local_pattern < 0)
	{
		local_pattern = 3;
	}
	
	stepping_motor_drive((unsigned char)local_pattern);
	
	return RET_SUCCESS;
}

int stepping_motor_drive(unsigned char pattern)
{
	
	//ステッピングモータの印加パターンを指定した位置に設定します
	
	//引数
	//  pattern
	//    0-3の印加パターン
	
	//戻り値
	//  RET_SUCCESS(0)
	
	//PB7 b3
	//PB6 b2
	//PB5 b1
	//PB4 b0
	
	unsigned char port = 0;
	
	switch(pattern)
	{
		case 0:
			//1010
			port = 0xA0;
			break;
			
		case 1:
			//1001
			port = 0x90;
			break;
			
		case 2:
			//0101
			port = 0x50;
			break;
		
		case 3:			
			//0110
			port = 0x60;
			break;
			
		default:
			break;
	}
	
	//PB7-PB4のみ変更
	PORTB.PODR.BYTE = (PORTB.PODR.BYTE & 0x0F) | port;
	
	return RET_SUCCESS;
}

int adc_start(void)
{
	//A/D変換の定期的な実行を開始します
	
	//引数
	//  なし
	
	//戻り値
	//  RET_SUCCESS(0)
	
	/*
	 * A/D変換は、CMT3のタイミングで変換開始指示を行います
	 * （ADCの結果は、A/D変換終了割り込みで回収）
	 */
	
	g_function_use[IOBOARD_ADC] = ENABLE;
	
	return RET_SUCCESS;
}

int adc_stop(void)
{
	//A/D変換の定期的な実行を停止します
	
	//引数
	//  なし
	
	//戻り値
	//  RET_SUCCESS(0)
	
	g_function_use[IOBOARD_ADC] = DISABLE;
	
	return RET_SUCCESS;
}

unsigned short adc_val(void)
{
	//A/D変換の結果を返します
	
	//引数
	//  なし
	
	//戻り値
	//  A/D変換値(0-0xFFF)
	
	/*A/D変換の結果は、移動平均を計算した値となります
	 */
	
	return g_adc_result;
}

int led7seg_init(void)
{
	//セグメントLED初期化関数
	
	//7セグLEDの初期化を行います
	
	//引数
	//  なし
	
	//戻り値
	//  RET_SUCCESS(0)
	
	PORTA.PODR.BYTE = 0x00;		//SEG信号線
	PORTA.PDR.BYTE = 0xFF;
	
	PORT7.PODR.BYTE = 0x00;		//COM信号線
	PORT7.PDR.BYTE = 0xF0;
	
	return RET_SUCCESS;
}

int led7seg_start(void)
{
	//7セグLEDの定期的な表示更新を開始します
	
	//引数
	//  なし
	
	//戻り値
	//  RET_SUCCESS(0)
	
	/*
	 * 7セグLEDの表示更新（COM信号を切り替える）は、CMT3のタイミングで行います
	 */
	
	g_function_use[IOBOARD_LED7SEG] = ENABLE;
	
	return RET_SUCCESS;
}

int led7seg_stop(void)
{
	//7セグLEDの定期的な表示更新を停止します
	
	//引数
	//  なし
	
	//戻り値
	//  RET_SUCCESS(0)
	
	g_function_use[IOBOARD_LED7SEG] = DISABLE;
	
	//表示は消灯
	PORTA.PODR.BYTE = 0x00;		//SEG信号線
	PORT7.PODR.BYTE = 0x00;		//COM信号線
	
	return RET_SUCCESS;
}

int led7seg_segument_data_set(unsigned char column, unsigned char seg, unsigned char seg_on_off)
{
	//セグメントLEDセグメント値変更関数
	
	//7セグLEDのセグメント単位でのデータ変更を行います
	//7セグLEDのセグメント制御のグローバル変数に結果が反映されます
	
	//任意の表示を行う際に使用
	
	//引数
	//  column : 1-4桁目の選択
	//  seg : データを変更するセグメント, 'A'-'G', 'P'(DP)
	//  seg_on_off : 0の場合は消灯, 1の場合は点灯 
	
	//戻り値
	//  正常終了 : RET_SUCCESS(0)
	//  引数エラーでグローバル変数の変更が行われなかった : RET_ERROR(-1)
	
	unsigned char seg_data;
	
	if (column < 1) return RET_ERROR;
	if (column > 4) return RET_ERROR;
	
	//操作するセグメントの指定
	switch(seg)
	{
		case 'A':
			seg_data = 0x01;	//SEG A -> PA0
			break;
		
		case 'B':
			seg_data = 0x02;	//SEG B -> PA1
			break;
	
		case 'C':
			seg_data = 0x04;	//SEG C -> PA2
			break;
		
		case 'D':
			seg_data = 0x08;	//SEG D -> PA3
			break;
		
		case 'E':
			seg_data = 0x10;	//SEG E -> PA4
			break;
		
		case 'F':
			seg_data = 0x20;	//SEG F -> PA5
			break;
	
		case 'G':
			seg_data = 0x40;	//SEG G -> PA6
			break;

		case 'P':
			seg_data = 0x80;	//SEG DP -> PA7
			break;
			
		default:
			return RET_ERROR;
			break;
	}
	
	switch(seg_on_off)
	{
		case 0:
			g_led7seg_data[column - 1] &= ~seg_data;		//データを変更する桁のセグメントデータのbitを0に変更
			break;
		
		case 1:
			g_led7seg_data[column - 1] |= seg_data;			//データを変更する桁のセグメントデータのbitを1に変更
			break;
		
		default:
			return RET_ERROR;
			break;
	}
	
	return RET_SUCCESS;
}

int led7seg_disp_hex(unsigned short data)
{
	//セグメントLED表示関数
	
	//7セグLEDの4桁をHEXデータで表示します
	//表示は0-9, A-Fのデータが対象です
	//7セグLEDのセグメント制御のグローバル変数に結果が反映されます
	
	//引数
	//  data : 表示データ
	
	//戻り値
	//  正常終了 : RET_SUCCESS(0)
	
	//表示例
	//data = 0x1234　-> 1234
	//data = 0xABCD  -> AbCd
	
	int i;
	
	for (i=0; i<4; i++)
	{
		switch ((data >> (i*4)) & 0xF)			//下位から4bit単位で切り出してデータ処理
		{
			case 0:
				g_led7seg_data[i] = 0x3F;		//0表示
				break;
				
			case 1:
				g_led7seg_data[i] = 0x06;		//1表示
				break;
		
			case 2:
				g_led7seg_data[i] = 0x5B;		//2表示
				break;
				
			case 3:
				g_led7seg_data[i] = 0x4F;		//3表示
				break;
				
			case 4:
				g_led7seg_data[i] = 0x66;		//4表示
				break;
				
			case 5:
				g_led7seg_data[i] = 0x6D;		//5表示
				break;
				
			case 6:
				g_led7seg_data[i] = 0x7D;		//6表示
				break;
				
			case 7:
				g_led7seg_data[i]= 0x07;		//7表示
				break;
				
			case 8:
				g_led7seg_data[i] = 0x7F;		//8表示
				break;
				
			case 9:
				g_led7seg_data[i] = 0x6F;		//9表示
				break;
				
			case 0xA:
				g_led7seg_data[i] = 0x77;		//A表示
				break;
				
			case 0xB:
				g_led7seg_data[i] = 0x7C;		//b表示
				break;
				
			case 0xC:
				g_led7seg_data[i] = 0x39;		//C表示
				break;
				
			case 0xD:
				g_led7seg_data[i] = 0x5E;		//d表示
				break;
				
			case 0xE:
				g_led7seg_data[i] = 0x79;		//E表示
				break;
				
			case 0xF:
				g_led7seg_data[i] = 0x71;		//F表示
				break;
				
			default:
				break;
		}
	}
	
	return RET_SUCCESS;
}

int led7seg_disp_num(short data, unsigned char dp)
{
	//セグメントLED表示関数
	
	//数値を7セグLEDで表示します
	//7セグLEDのセグメント制御のグローバル変数に結果が反映されます
	
	//引数
	//  data : 表示データ(-999 - 9999 が有効)
	//  dp : 小数点表示(0-4が有効, 1は最下位桁に小数点表示, 0は小数点表示しない）
	
	//戻り値
	//  正常終了 : RET_SUCCESS(0)
	//  引数エラーでグローバル変数の変更が行われなかった : RET_ERROR(-1)
	
	//表示例
	//data = 1234, dp=2 -> 123.4
	
	int i;
	short abs_num;
	unsigned char num[4];
	
	if (data < -999) return RET_ERROR;
	if (data > 9999) return RET_ERROR;
	
	if (dp > 4) return RET_ERROR;
	
	//桁毎のデータに分解
	abs_num = data;
	if (data < 0)
	{
		abs_num *= -1;	//負数の場合は絶対値を取る
	}
	
	num[3] = (unsigned char)(abs_num / 1000);
	abs_num -= num[3] * 1000;
	num[2] = (unsigned char)(abs_num / 100);
	abs_num -= num[2] * 100;
	num[1] = (unsigned char)(abs_num / 10);
	abs_num -= num[1] * 10;
	num[0] = (unsigned char)abs_num;
	
	//この時点でnum[]には
	//data = 12のとき
	//num[3] = 0
	//num[2] = 0
	//num[1] = 1
	//num[0] = 2
	//が入っている
	
	//0012の時は__12（_は空白）と表示したい
	//最上位桁からスキャンして0であれば、空白表示に置き換え、0以外であればスキャンを終了（それ以降の値は表示）
	//最下位桁は常に表示
	for (i=3; i>=1; i--)
	{
		if (num[i] == 0)
		{
			num[i] = 0xFF;		//空白表示
		}
		else
		{
			//0以外の値であればスキャンを終了
			//1001の場合、途中の0は表示対象
			break;
		}
	}	
	
	//負数の場合は最下位桁からスキャンして最初に見つかった空白を'-'に置き換え
	if (data < 0)
	{
		for (i=0; i<4; i++)
		{
			if (num[i] == 0xff)
			{
				num[i] = 0x80;			//'-'表示
				break;
			}
		}
	}
	
	for (i=0; i<4; i++)
	{
		switch (num[i])
		{
			case 0:
				g_led7seg_data[i] = 0x3F;		//0表示
				break;
				
			case 1:
				g_led7seg_data[i] = 0x06;		//1表示
				break;
		
			case 2:
				g_led7seg_data[i] = 0x5B;		//2表示
				break;
				
			case 3:
				g_led7seg_data[i] = 0x4F;		//3表示
				break;
				
			case 4:
				g_led7seg_data[i] = 0x66;		//4表示
				break;
				
			case 5:
				g_led7seg_data[i] = 0x6D;		//5表示
				break;
				
			case 6:
				g_led7seg_data[i] = 0x7D;		//6表示
				break;
				
			case 7:
				g_led7seg_data[i]= 0x07;		//7表示
				break;
				
			case 8:
				g_led7seg_data[i] = 0x7F;		//8表示
				break;
				
			case 9:
				g_led7seg_data[i] = 0x6F;		//9表示
				break;
				
			case 0xFF:
				g_led7seg_data[i] = 0x00;		//空白
				break;
				
			case 0x80:
				g_led7seg_data[i] = 0x40;		//SEG G のみ点灯 '-'表示
				break;
				
			default:
				break;
		}
	}
	
	//小数点の表示
	if (dp != 0)
	{
		led7seg_segument_data_set(dp, 'P', 1);		//DPのセグメントを追加表示
	}
	
	return RET_SUCCESS;
}

int matrixsw_init(void)
{
	//マトリックススイッチ初期化関数
	
	//SW11-SW26の初期化を行います
	
	//引数 なし
	
	//戻り値
	//  RET_SUCCESS(0)
	
	PORT6.PODR.BYTE = 0xF0;		//P64-P67=H
	PORT6.PDR.BYTE = 0xF0;		//P64-P67を出力、P60-P63は入力に設定
	
	return RET_SUCCESS;
}

int matrixsw_start(void)
{
	//マトリックススイッチの定期的な読み取りを開始します
	
	//引数
	//  なし
	
	//戻り値
	//  RET_SUCCESS(0)
	
	/*
	 * マトリックススイッチの読み取り(1/4ずつ）は、CMT3のタイミングで行います
	 */
	
	g_function_use[IOBOARD_MATRIXSW] = ENABLE;
	
	return RET_SUCCESS;
}

int matrixsw_stop(void)
{
	//マトリックススイッチの定期的な読み取りを停止します
	
	//引数
	//  なし
	
	//戻り値
	//  RET_SUCCESS(0)
	
	g_function_use[IOBOARD_MATRIXSW] = DISABLE;
	
	return RET_SUCCESS;
}

int matrixsw_read(unsigned char key)
{
	//マトリックススイッチの読み出しを行います
	
	//引数
	//  key 0: SW11, 1:SW12, ...15:SW26
	
	//戻り値
	//  0 : SWが押されている
	//  1 : SWが押されていない
	//  引数エラー : RET_ERROR(-1)
	
	if (key > 0xf)
	{
		return RET_ERROR;
	}
	
	return (int)g_matrixsw[key];
}
