// lcd_1602.c
//
// SC1602 charactor LCD operation routine
//
// 4bit connection
//

/*
 * @file  lcd_1602.c
 * @brief LCDドライバ
 *
 * 本ソフトウェアの著作権は作成元である(株)北斗電子が所有するものとし、
 * (株)北斗電子は、以下の (1)～(3) の条件を満たす場合に限り、
 * 本ソフトウェア（本ソフトウェアを改変したものを含む。以下同じ）を
 * 使用・複製・改変・再配布（以下、利用と呼ぶ）することを無償で許諾する。
 *
 * (1) 本ソフトウェアをソースコードの形で利用する場合には、下記の著作
 *	   権表示、この利用条件が、そのままの形でソースコード中に含まれて
 *	   いること。
 * (2) 本ソフトウェアの一部または全てを無断で転載することを禁止するもの
 *	   とする。雑誌などへ紹介・収録の場合は(株)北斗電子に連絡願います。
 * (3) 本ソフトウェアの利用により直接的または間接的に生じるいかなる損害
 *	   からも、(株)北斗電子は一切の責任を負わないものとする。
 *
 * Copyright (C) Hokuto denshi Co,Ltd. 2018-2022
 */

#include <machine.h>
#include "r_smc_entry.h"
#include "lcd_1602.h"
#include "lcd_1602_RX65.h"

//関数定義（ローカル）
void lcd_wait_msec(unsigned long msec);
void lcd_wait_s(void);
void lcd_enable_rise(void);
void lcd_enble_fall(void);
void lcd_rs_rise(void);
void lcd_rs_fall(void);
void lcd_db_set(unsigned char c4);
void lcd_port_init (void);

const unsigned char table_h[] = {"0123456789ABCDEF"};

//msec ウェイト関数
void lcd_wait_msec(unsigned long msec)
{
	volatile unsigned long lx;
	volatile unsigned short sx;
	
	for(lx=0; lx<msec; lx++)
	{
		for(sx=0; sx<50; sx++) lcd_wait_s();
	}
}

//ウェイト関数 (20us)
void lcd_wait_s(void)
{
	volatile unsigned short sx;
	
	for(sx=0; sx<WAIT_20US; sx++) nop();
}

//E(Enable) rise関数
void lcd_enable_rise(void)
{
	LCD_1602_E_PODR = 1;
	nop(); nop(); nop(); nop();
}

//E(Enable) fall関数
void lcd_enable_fall(void)
{
	LCD_1602_E_PODR = 0;
	nop(); nop(); nop(); nop();
}

//RS rise関数
void lcd_rs_rise(void)
{
	LCD_1602_RS_PODR = 1;
	nop(); nop(); nop(); nop();
}

//RS fall関数
void lcd_rs_fall(void)
{
	LCD_1602_RS_PODR = 0;
	nop(); nop(); nop(); nop();
}

//信号送出関数
void lcd_db_set (unsigned char c4)
{
	//DBビットを送出する関数
	//引数
	//  unsigned char c4 : 0x0 - 0xf のコードを受け取る
	
	LCD_1602_DB7_PODR = (c4 & 0x8) >> 3;
	LCD_1602_DB6_PODR = (c4 & 0x4) >> 2;
	LCD_1602_DB5_PODR = (c4 & 0x2) >> 1;
	LCD_1602_DB4_PODR = c4 & 0x1;
	
}

//コマンド送信関数
void lcd_cmd(unsigned char c)
{
	//任意のコマンドを送信する関数
	//引数
	// unsigned char chara : 制御コード（8bit)
	
	//RS=0
	lcd_rs_fall();
	
	//upper
	//E->H
	lcd_enable_rise();
	//DB7, DB6, DB5, DB4 = chara upper 4bit
	lcd_db_set(c>>4);
	lcd_wait_s();
	//E->L
	lcd_enable_fall();
	lcd_wait_s();
	
	//lower
	//E->H
	lcd_enable_rise();
	//DB3, DB2, DB1, DB0 = chara lower 4bit
	lcd_db_set(c & 0xf);
	lcd_wait_s();
	//E->L
	lcd_enable_fall();
	lcd_wait_s();
}

//LCD初期化関数
void lcd_init (void)
{
	
	unsigned char i;
	
	lcd_port_init();
	
	lcd_wait_msec(40);	//5V:15ms, 3V:40msウェイトが必要
	
	lcd_rs_fall();
	
	//8bitモード切替 DB7=0, DB6=0, DB5=1, DB4=1
	lcd_db_set(0x3);
	lcd_wait_s();
	
	for(i=0; i<3; i++)		//どんな状態からでも8bitモードに切り替えできる様3回繰り返しで入れる
	{
		lcd_enable_rise();
		lcd_wait_s();
		lcd_enable_fall();
		lcd_wait_msec(5);	//4.1ms以上
	}
	
	//function set(4bit mode)
	//E->H
	lcd_enable_rise();
	//DB7=0, DB6=0, DB5=1, DB4=0
	lcd_db_set(0x2);
	lcd_wait_s();
	//E->L
	lcd_enable_fall();
	
	lcd_wait_msec(5);
	
	//2line mode
	lcd_cmd(0x28);		//2行表示
	
	//b2 D:文字表示ON, b1 C:下線カーソルON, b0 B:ブロックカーソルON
	lcd_cmd(0x0c);				//画面表示ON, カーソルOFF
	//lcd_cmd(0x0c | 0x01);		//ブロックカーソルON	
	//lcd_cmd(0x0c | 0x02);		//下線カーソルON
	//lcd_cmd(0x0c | 0x03);		//ブロックカーソル＋下線カーソルON
	
	//display->CLEAR
	lcd_cmd(0x01);
	lcd_wait_msec(2);	//1.6ms以上
	
	//entry mode		//b1 I/D 0:デクリメント, 1:インクリメント, b0 S:表示シフトあり
	lcd_cmd(0x06);
}

void lcd_port_init (void)
{
	MPC.PWPR.BIT.B0WI  = 0;
	MPC.PWPR.BIT.PFSWE = 1;
	
	LCD_1602_RS_PFS
	LCD_1602_E_PFS
	LCD_1602_DB4_PFS
	LCD_1602_DB4_PFS
	LCD_1602_DB4_PFS
	LCD_1602_DB4_PFS
	
	MPC.PWPR.BIT.PFSWE = 0;
	
	LCD_1602_RS_PMR
	LCD_1602_E_PMR
	LCD_1602_DB4_PMR
	LCD_1602_DB5_PMR
	LCD_1602_DB6_PMR
	LCD_1602_DB7_PMR
	
	LCD_1602_RS_PODR  = 0;
	LCD_1602_E_PODR   = 0;
	LCD_1602_DB4_PODR = 0;
	LCD_1602_DB5_PODR = 0;
	LCD_1602_DB6_PODR = 0;
	LCD_1602_DB7_PODR = 0;
	
	LCD_1602_RS_PDR  = 1;
	LCD_1602_E_PDR   = 1;
	LCD_1602_DB4_PDR = 1;
	LCD_1602_DB5_PDR = 1;
	LCD_1602_DB6_PDR = 1;
	LCD_1602_DB7_PDR = 1;
}

void lcd_hs1 (void)
{
	//1行目の1文字目にカーソルを移動する関数
	
	lcd_cmd(0x80);
	lcd_wait_s();
	lcd_wait_s();
	
}

void lcd_hs2 (void)
{
	//2行目の1文字目にカーソルを移動する関数
	
	lcd_cmd(0xc0);
	lcd_wait_s();
	lcd_wait_s();
	
}

void lcd_clear (void)
{
	//ディスプレイをクリアする関数
	
	lcd_cmd(0x01);
	lcd_wait_msec(2);
}

/* LCD表示関数 */
void lcd_write_char(unsigned char c)
{
	//任意の1文字を表示する関数
	//引数
	// unsigned char chara : 文字コード（8bit)
	
	//RS=1
	lcd_rs_rise();
	
	//upper
	//E->H
	lcd_enable_rise();
	//DB7, DB6, DB5, DB4 = chara upper 4bit
	lcd_db_set(c>>4);
	lcd_wait_s();
	//E->L
	lcd_enable_fall();
	lcd_wait_s();
	
	//lower
	//E->H
	lcd_enable_rise();
	//DB3, DB2, DB1, DB0 = chara lower 4bit
	lcd_db_set(c & 0xf);
	lcd_wait_s();
	//E->L
	lcd_enable_fall();
	lcd_wait_s();
}

void lcd_write_hex(unsigned char c)
{
	//与えられた文字をhex2桁(00-ff)で表示する
	//引数
	// unsigned char chara : 文字コード（8bit)
	
	lcd_write_char(table_h[(c >> 4)]);
	lcd_write_char(table_h[(c & 0xf)]);
}

void lcd_write_byte_int( unsigned char num )
{
	unsigned char n[3];
	
	n[0]  = num % 10;
	num  /= 10;
	n[1]  = num % 10;
	num  /= 10;
	n[2]  = num % 10;
	
	if(n[2] != 0)
	{
		lcd_write_char(table_h[n[2]]);
		lcd_write_char(table_h[n[1]]);
	}
	else if(n[1] != 0)
	{
		lcd_write_char(table_h[n[1]]);
	}

	lcd_write_char(table_h[n[0]]);
		
}

void lcd_write_short_int( unsigned short num )
{
	unsigned char n[5];
	
	n[0]  = num % 10;
	num  /= 10;
	n[1]  = num % 10;
	num  /= 10;
	n[2]  = num % 10;
	num  /= 10;
	n[3]  = num % 10;
	num  /= 10;
	n[4]  = num % 10;
	
	if(n[4] != 0)
	{
		lcd_write_char(table_h[n[4]]);
		lcd_write_char(table_h[n[3]]);
		lcd_write_char(table_h[n[2]]);
		lcd_write_char(table_h[n[1]]);
	}
	else if(n[3] != 0)
	{
		lcd_write_char(table_h[n[3]]);
		lcd_write_char(table_h[n[2]]);
		lcd_write_char(table_h[n[1]]);
	}
	else if(n[2] != 0)
	{
		lcd_write_char(table_h[n[2]]);
		lcd_write_char(table_h[n[1]]);
	}
	else if(n[1] != 0)
	{
		lcd_write_char(table_h[n[1]]);
	}

	lcd_write_char(table_h[n[0]]);
		
}

void lcd_write_str(unsigned char *str)
{

	//文字列表示関数
	//  引数
	//    unsigned char *str: 表示させたい文字列のポインタ（'\0'終端）
	
	while(*str != '\0')
	{
	  lcd_write_char(*str);
	  str++;
	}
}

void lcd_cursor_move(unsigned char lines, unsigned char pos)
{
	//カーソル移動関数
	//  引数
	//    unsigned char lines: 行(1 or 2)
	//    unsigned char pos: カーソル位置
	
	unsigned char cmd = 0x80;		//カーソル移動コマンド
	
	if (lines == 2) cmd |= 0x40;	//2行目はカーソル位置0x40から始まる
	
	cmd += pos;
	
	lcd_cmd(cmd);
}
