/*
 * @file  main.c
 * @brief HSBRX65_IO_BOARD サンプルプログラムメイン関数
 *
 * 本ソフトウェアの著作権は作成元である(株)北斗電子が所有するものとし、
 * (株)北斗電子は、以下の (1)-(3) の条件を満たす場合に限り、
 * 本ソフトウェア（本ソフトウェアを改変したものを含む。以下同じ）を
 * 使用・複製・改変・再配布（以下、利用と呼ぶ）することを無償で許諾する。
 *
 * (1) 本ソフトウェアをソースコードの形で利用する場合には、下記の著作
 *	   権表示、この利用条件が、そのままの形でソースコード中に含まれて
 *	   いること。
 * (2) 本ソフトウェアの一部または全てを無断で転載することを禁止するもの
 *	   とする。雑誌などへ紹介・収録の場合は(株)北斗電子に連絡願います。
 * (3) 本ソフトウェアの利用により直接的または間接的に生じるいかなる損害
 *	   からも、(株)北斗電子は一切の責任を負わないものとする。
 *
 * Copyright (C) Hokuto denshi Co,Ltd. 2024
 */

/*----------------------------------------------------------------------
	インクルード
----------------------------------------------------------------------*/
#include	"r_smc_entry.h"

#include	"main.h"
#include	"../io_board/io_board.h"
#include	"../lcd_1602/lcd_1602.h"
#include	"../sci/sci.h"

/*----------------------------------------------------------------------
	定数定義
----------------------------------------------------------------------*/
#define LED7SEG_UPDATE_RATE	(1000)		//7セグLEDのA/D値が変化しない場合は、概ね1000ms毎に表示値を更新する

/*----------------------------------------------------------------------
	グローバル変数（ファイル内）
----------------------------------------------------------------------*/
static volatile int g_function = 0;
static volatile unsigned short g_sw10_toggle = 0;
static volatile unsigned long g_cmt3_wait_counter;
static volatile int g_stepping_motor_direction = 0;		//ステッピングモータの回転方向, 0:停止, 1:正転, -1:逆転
static volatile unsigned short g_local_adc_val;			//一定のタイミングで取得したA/D変換値

/*----------------------------------------------------------------------
	関数プロトタイプ宣言
----------------------------------------------------------------------*/
void main_etc(void);
void wait_ms(unsigned long ms);

//割り込みコールバック関数
void cmt3_callback_etc(void);
void rtc_callback_etc(void);
void adc_callback_etc(void);
void sw8_callback_etc(void);
void sw9_callback_etc(void);
void sw10_callback_etc(void);
void tpu2a_callback_etc(void);
void tpu2b_callback_etc(void);

/*----------------------------------------------------------------------
	関数定義
----------------------------------------------------------------------*/
void main_etc(void)
{
	
	//I/Oボードの各種機能を紹介するサンプルプログラム
	
	int current_function = -1;				//現在の動作モード（ダミー）
	
	unsigned char *func_name[6];			//使用機能表示
	
	int buzzer_enable = 0;					//ブザーを鳴らす
	int led8_enable = 0;					//LED8の点滅
	int stepping_motor_enable = 0;			//ステッピングモータを回す
	unsigned short led7seg_disp;			//7セグLED表示値
	unsigned long led7seg_disp_counter;		//7セグLEDの表示更新カウンタ
	unsigned short freq;					//ブザーの周波数
	unsigned long cycle;					//LED8の周期
	float duty;								//LED8のduty比
	unsigned short stepping_motor_rpm;		//ステッピングモータの回転数
	char buf[20];							//文字表示用バッファ
	int i;
	
	//割り込みコールバック関数の割り当て
	cmt3_callback = cmt3_callback_etc;
	rtc_callback = rtc_callback_etc;
	adc_callback = adc_callback_etc;
	sw8_callback = sw8_callback_etc;
	sw9_callback = sw9_callback_etc;
	sw10_callback = sw10_callback_etc;
	tpu2a_callback = tpu2a_callback_etc;
	tpu2b_callback = tpu2b_callback_etc;
	
	//機能説明（LCD上の表示）
	//              1234567890123456
	func_name[0] = "SW0-7 -> LED0-7 ";
	func_name[1] = "MATRIX-SW -> LCD";
	func_name[2] = "VR -> 7SEG LED  ";
	func_name[3] = "BUZZER(VR->FREQ)";
	func_name[4] = "LED8(SW10:CY/DT)";
	func_name[5] = "STEPPING MOTOR  ";
	
	//各種初期化
	push_sw_init();				//プッシュスイッチ(SW0-SW7)初期化
	intr_push_sw_init(0);		//0:スイッチを押した際に反応する様にする
	led7seg_init();				//7セグLEDの初期化
	matrixsw_init();			//マトリックススイッチの初期化
	led_init();					//LED(LED0-LED7)初期化
	stepping_motor_init();		//ステッピングモータ初期化
	lcd_init();					//LCD初期化
	
	//動作スタート
	intr_push_sw_start();		//割り込みスイッチ動作開始
	R_Config_CMT3_Start();		//ADC, 7セグLED, マトリックススイッチのタイマ動作開始
	adc_start();				//A/D変換動作開始
	led7seg_start();			//7セグLED動作
	matrixsw_start();			//マトリックススイッチ読み取り
	
	//J7(USB-miniB)への情報表示(115,200bps)
	sci_start();
	sci_write_str("\n\nCopyright (C) 2024 HokutoDenshi. All Rights Reserved.\n");
	sci_write_str("HSBRX65-IO-BOARD SAMPLE PROGRAM.\n\n");
	
	sci_write_str("Board function demo.\n");
	
	//LCDへの情報表示
	lcd_hs1();
	//             1234567890123456
	lcd_write_str("HSBRX65-IO-BOARD");
	lcd_hs2();
	lcd_write_str("SAMPLE PROGRAM2");
	wait_ms(3000);							//LCD表示確認のため3秒停止
	
	/*
	 * SW8  : 機能選択UP
	 * SW9  : 機能選択DOWN
	 * SW10 : 機能内パラメータ設定
	 */
	
	while(1)
	{
		if (current_function != g_function)
		{
			
			//使用機能が変わったので既存動作のものを止める
			if (buzzer_enable == ENABLE)
			{
				buzzer_off();
				buzzer_enable = DISABLE;
			}
			
			if (led8_enable == ENABLE)
			{
				led8_off();
				led8_enable = DISABLE;
			}
			
			if (stepping_motor_enable == ENABLE)
			{
				g_stepping_motor_direction = 0;
				stepping_motor_stop();				//出力をディスエーブル制御
				stepping_motor_enable= DISABLE;
			}
			
			//使用機能が変わったタイミングでLCDの1行目に機能表示
			lcd_hs1();
			lcd_write_str(func_name[g_function]);
			
			//LCD2行目の表示
			lcd_hs2();
			//             1234567890123456
			lcd_write_str("                ");

			current_function = g_function;	//現在選択中の機能を保存
		}
		
		switch (g_function)
		{
			case 0:
				//SW0-7 -> LED0-7が連動する動作
				led_set(~push_sw_read());
				break;
			
			case 1:
				//マトリックススイッチの読み取り結果がLCDの2行目に表示される動作
				//-:押していない
				//*:押している
				lcd_hs2();
				for (i=0; i<16; i++)
				{
					if (matrixsw_read((unsigned char)i) == 0)
					{
						lcd_write_char('*');	//SWが押されている場合
					}
					else
					{
						lcd_write_char('-');	//SWが押されていない場合
					}
				}
				break;
				
			case 2:
				//ボリューム(VR, R29)のA/D変換結果が7セグメントLEDに表示される動作
				if ((led7seg_disp >= (adc_val() + 3)) || (led7seg_disp <= (adc_val() - 3)) || (led7seg_disp_counter == 0))
				{
					//数値が3以上変化した場合,または1秒経過後は直ぐに表示値を更新
					led7seg_disp = adc_val();
					led7seg_disp_counter = LED7SEG_UPDATE_RATE;
				}
				else
				{
					
					led7seg_disp_counter--;
				}
				
				led7seg_disp_num(4095 - led7seg_disp, 0);	//0は小数点を表示しない, 時計回り->表示値増加とする（0-4095の10進数表示）
				wait_ms(1);
				break;
				
			case 3:
				//ブザーを鳴らす, ボリュームで音の高さを変える
				if (buzzer_enable == DISABLE)
				{
					buzzer_on();
					buzzer_enable = ENABLE;
				}
				//VR->ブザーの周波数を変える
				//VR 4095:300Hz, 0:8000Hz
				freq = (4095 - adc_val()) * (8000-300) / 4096 + 300;
				buzzer_freq_set(freq);
				lcd_hs2();
				//1234567890123456
				//FREQ:8000[Hz]   
				lcd_write_str("FREQ:");
				lcd_write_short_int(freq);
				lcd_write_str("[Hz]     ");
				break;
				
			case 4:
				//LED8の点滅
				//VR->LEDの点滅周期の変更 or LEDの点滅dutyの変更（SW10で切り換え）
				if (led8_enable == DISABLE)
				{
					led8_on();
					led8_duty_set(0.5f);	//初期値はduty=50%にする
					led8_enable = ENABLE;
					g_sw10_toggle = 0;		//SW10は周期とduty設定の切り替え
				}
				
				if ((g_sw10_toggle % 2) == 0)
				{
					//VRで周期の変更
					cycle = (int)((float)(4095 - g_local_adc_val) * 1.001f) * 65536 / 4096 + 1;	//16bitの範囲の周期設定（最低1）※TPU2の周期で取得したA/D値を使用
					/*
					 * cycle 1-65535 の範囲（フルスケール）で設定したいので、VR設定値を0.1%増し（4095/4096の補正）で計算する
					 */
					if (cycle > 65535) cycle = 65535;
					led8_cycle_set((unsigned short)cycle);
					lcd_hs2();
					//1234567890123456
					//CYCLE:12345     
					lcd_write_str("CYCLE:");
					lcd_write_short_int(cycle);
					lcd_write_str("         ");
				}
				else
				{
					//VRでdutyの変更
					duty = (float)(4095 - adc_val()) * 1.001f / 4096.0f;
					/*
					 * duty 0-1 の範囲（フルスケール）で設定したいので、VR設定値を0.1%増し（4095/4096の補正）で計算する
					 * ※dutyが1(100%)になっても、表示上のみで、実際は出力波形はL張り付きにはならない
					 */
					if (duty > 1.0f) duty = 1.0f;
					led8_duty_set(duty);
					float2str(duty * 100.0f, 1, buf);	//duty値を[%]で小数点以下1桁まで文字列変換
					lcd_hs2();
					//1234567890123456
					//DUTY:25.1[%]
					lcd_write_str("DUTY:");
					lcd_write_str((unsigned char *)buf);
					lcd_write_str("         ");
				}
				break;
				
			case 5:
				//ステッピングモータの駆動
				//VR->回転速度 & 回転方向（センター付近で停止, 回転軸に応じた方向、速度設定）
				if (stepping_motor_enable == DISABLE)
				{
					led8_on();							//LED8のタイマ(TPU2)でステッピングモータを次のステップへ動かす
					led8_enable = ENABLE;
					stepping_motor_enable = ENABLE;
				}
				//VR 逆回転---------|---------正回転
				//   4095         2048          0
				

				if (g_adc_result < 2048)
				{
					//正回転
					g_stepping_motor_direction = 1;
					stepping_motor_rpm = (unsigned short)((float)(2048 - adc_val()) * 8.0f / 2048.0f + 0.5f);	//8ステップに変換, 計算結果を四捨五入
				}
				else
				{
					//逆回転
					g_stepping_motor_direction = -1;
					stepping_motor_rpm = (unsigned short)((float)(adc_val() - 2048) * 8.0f / 2048.0f + 0.5f);	//8ステップに変換, 計算結果を四捨五入
				}
				
				if (stepping_motor_rpm == 0)
				{
					//VRがセンタ付近の時はステッピングモータを動かさない
					g_stepping_motor_direction = 0;
					led8_duty_set(0.0f);		//LED8は消灯
				}
				else
				{
					//stepping_motor_rpm=1の時は約1秒に1回のステップで動かす
					//ステッピングモータはTPU2(LED8)と連動させて動かす
					cycle = (unsigned short) (1.0f / (float)(stepping_motor_rpm) / (1.0f / (PCLKB * 1.0e6f) * 1024));
					led8_cycle_set(cycle);
					led8_duty_set(0.1f);		//dutyは10%（ステッピングモータが動く際にLED8が一瞬光る）
				}
				
				lcd_hs2();
				
				
				if (g_stepping_motor_direction == 0)
				{
					             //12345
					lcd_write_str("STOP ");
				}
				else if (g_stepping_motor_direction == 1)
				{
					lcd_write_str("RIGHT");
				}
				else
				{
					lcd_write_str("LEFT ");
				}
				
				//             6789012
				lcd_write_str(" SPEED=");
				lcd_write_byte_int((unsigned char)stepping_motor_rpm);
				lcd_write_str("     ");
				break;
		}
	}
}

void wait_ms(unsigned long ms)
{
	//ウェイト関数
	
	//指定した時間ウェイトを掛けます
	
	//引数
	//  ms : 待ち時間[ms]
	
	//戻り値
	//  なし
	
	g_cmt3_wait_counter = ms;
	
	while (g_cmt3_wait_counter != 0)
	{
		__nop();
	}
}

/*----------------------------------------------------------------------
	割り込みコールバック関数
----------------------------------------------------------------------*/
void cmt3_callback_etc(void)
{
	//wait_ms()関数向けカウンタ操作
	if (g_cmt3_wait_counter != 0)
	{
		g_cmt3_wait_counter--;
	}
}

void rtc_callback_etc(void){}
void adc_callback_etc(void){}

void sw8_callback_etc(void)
{
	//SW8が押されると次の機能に移動
	
	g_function++;
	
	if (g_function > 5)
	{
		g_function = 0;		//1周すると先頭に戻る
	}
}

void sw9_callback_etc(void)
{
	//SW9が押されると前の機能に移動
	
	g_function--;
	
	if (g_function < 0)
	{
		g_function = 5;		//1周すると最後に移動
	}
}

void sw10_callback_etc(void)
{
	//SW10が押されると機能により設定項目を変更
	
	g_sw10_toggle++;
}

void tpu2a_callback_etc(void)
{
	//ステッピングモータを1ステップ動かす
	
	if (g_stepping_motor_direction != 0)
	{
		stepping_motor_step_move(g_stepping_motor_direction);
	}
	
	//TPU2の周期に連動したA/D変換値
	g_local_adc_val = adc_val();
}

void tpu2b_callback_etc(void){}
