/*
 * @file  main.c
 * @brief HSBRX65_IO_BOARD サンプルプログラムメイン関数
 *
 * 本ソフトウェアの著作権は作成元である(株)北斗電子が所有するものとし、
 * (株)北斗電子は、以下の (1)-(3) の条件を満たす場合に限り、
 * 本ソフトウェア（本ソフトウェアを改変したものを含む。以下同じ）を
 * 使用・複製・改変・再配布（以下、利用と呼ぶ）することを無償で許諾する。
 *
 * (1) 本ソフトウェアをソースコードの形で利用する場合には、下記の著作
 *	   権表示、この利用条件が、そのままの形でソースコード中に含まれて
 *	   いること。
 * (2) 本ソフトウェアの一部または全てを無断で転載することを禁止するもの
 *	   とする。雑誌などへ紹介・収録の場合は(株)北斗電子に連絡願います。
 * (3) 本ソフトウェアの利用により直接的または間接的に生じるいかなる損害
 *	   からも、(株)北斗電子は一切の責任を負わないものとする。
 *
 * Copyright (C) Hokuto denshi Co,Ltd. 2024
 */

/*----------------------------------------------------------------------
	インクルード
----------------------------------------------------------------------*/
#include	"r_smc_entry.h"

#include	"main.h"
#include	"../io_board/io_board.h"
#include	"../lcd_1602/lcd_1602.h"
#include	"../sci/sci.h"

/*----------------------------------------------------------------------
	定数定義
----------------------------------------------------------------------*/
#define BUZZER_PATTERN_CYCLE	(10)
#define TIMER_MAX_SEC			(1000)	//タイマは1000秒までの時間を設定
#define TIMER_NONE				(0)		//タイマ設定なし
#define TIMER_VAL_SETTING		(1)		//タイマ値設定中
#define TIMER_COUNT				(2)		//タイマカウント中
#define TIMER_ALARM_1			(1)		//タイマアラーム鳴動中（ピピ　ピピ）
#define TIMER_ALARM_2			(2)		//タイマアラーム鳴動中（連続音）
#define TIMER_ALARM_1_TIME		(30)	//タイマは30秒で連続音に移行

/*----------------------------------------------------------------------
	構造体
----------------------------------------------------------------------*/

/*----------------------------------------------------------------------
	グローバル変数（ファイル内）
----------------------------------------------------------------------*/
static volatile int g_lcd_update = 1;			//LCDの2行目の更新タイミング（初回は無条件で更新）
static volatile int g_buzzer_flow = 0;			//0:TPU2割り込みで何もしない, 1:パターンで鳴らす, 2:dutyで鳴らす
static volatile unsigned char g_buzzer_pattern[BUZZER_PATTERN_CYCLE];
static volatile int g_sw8_push = 0;
static volatile int g_sw9_push = 0;
static volatile int g_sw10_push = 0;
//曜日
static unsigned char *dayw[7] = {"SUN", "MON", "TUE", "WED", "THU", "FRI", "SAT"}; 

static volatile int g_timer_operation = 0;		//タイマ設定, 0:設定なし, 1:タイマ値変更中, 2:タイマカウント中, 3:アラーム鳴動中(1), 4:アラーム鳴動中(2) 
static volatile int g_timer_val = 0;			//タイマ設定値

/*----------------------------------------------------------------------
	関数プロトタイプ宣言
----------------------------------------------------------------------*/
void main_timer(void);
void alarm_on(int level);
void alarm_off(void);
void calender_set(void);
void calender_set_lcd_disp(int process, int *modify);
int day_of_week(int y, int m, int d);
void led7seg_off(void);

//割り込みコールバック関数
void cmt3_callback_timer(void);
void rtc_callback_timer(void);
void adc_callback_timer(void);
void sw8_callback_timer(void);
void sw9_callback_timer(void);
void sw10_callback_timer(void);
void tpu2a_callback_timer(void);
void tpu2b_callback_timer(void);

/*----------------------------------------------------------------------
	関数定義
----------------------------------------------------------------------*/
void main_timer(void)
{
	//I/Oボードを使用してタイマ動作を行うサンプルプログラム
	
	//カレンダ表示値
	rtc_calendarcounter_value_t disp;
	
	unsigned char day_cnt = 0x00;		//日付が変わった際LCDの1行目を更新, 初期値は存在しない日付とする
	int timer_alarm = 0;				//アラームの状態 0:停止, 1:鳴動, 2:連続音
	int vr_invalid = 0;					//アラームの時間設定 0:VRで調整, 1:SW9, SW10で微調整
	int current_adc_val;				//A/D変換値を保存, 大きく変わった場合はVRを再度有効化する
	
	//割り込みコールバック関数の割り当て
	cmt3_callback = cmt3_callback_timer;
	rtc_callback = rtc_callback_timer;
	adc_callback = adc_callback_timer;
	sw8_callback = sw8_callback_timer;
	sw9_callback = sw9_callback_timer;
	sw10_callback = sw10_callback_timer;
	tpu2a_callback = tpu2a_callback_timer;
	tpu2b_callback = tpu2b_callback_timer;
	
	//各種初期化
	push_sw_init();				//プッシュスイッチ(SW0-SW7)初期化
	intr_push_sw_init(0);		//0:スイッチを押した際に反応する様にする
	led7seg_init();				//7セグLEDの初期化
	matrixsw_init();			//マトリックススイッチの初期化
	led_init();					//LED(LED0-LED7)初期化
	stepping_motor_init();		//ステッピングモータ初期化
	lcd_init();					//LCD初期化
	
	//動作スタート
	intr_push_sw_start();		//割り込みスイッチ動作開始
	R_Config_CMT3_Start();		//ADC, 7セグLED, マトリックススイッチのタイマ動作開始
	R_Config_RTC_Start();		//リアルタイムクロック動作開始
	adc_start();				//A/D変換動作開始
	led7seg_start();			//7セグLED動作
	matrixsw_start();			//マトリックススイッチ読み取り

	//J7(USB-miniB)への情報表示(115,200bps)
	sci_start();
	sci_write_str("\n\nCopyright (C) 2024 HokutoDenshi. All Rights Reserved.\n");
	sci_write_str("HSBRX65-IO-BOARD TIMER SAMPLE PROGRAM.\n\n");
	
	/*
	 * SW8  : カレンダ（時計）変更, 年→月→日→時間→分→秒→設定終了
	 * SW9  : 日付や時間を戻す
	 * SW10 : 日付や時間を進める
	 */
	 
	 /*
	  * SW0 : タイマセット
	  * SW1 : タイマスタート
	  * SW7 : タイマキャンセル
	  * SW0-SW7 : アラーム停止
	  */
	
	while(1)
	{	
		//LCD表示更新
		if (g_lcd_update == 1)
		{
			R_Config_RTC_Get_CalendarCounterValue(&disp);
			
			//2行目
			//1234567890123456
			//20:25:30
			lcd_hs2();
			lcd_write_hex(disp.rhrcnt);
			lcd_write_char(':');
			lcd_write_hex(disp.rmincnt);
			lcd_write_char(':');
			lcd_write_hex(disp.rseccnt);
			
			if (disp.rdaycnt != day_cnt)	//日付が変わったタイミング
			{
				//1行目
				//1234567890123456
				//2024/12/31 (FRY)
				lcd_hs1();
				lcd_write_str("20");
				lcd_write_hex(disp.ryrcnt);
				lcd_write_char('/');
				lcd_write_hex(disp.rmoncnt);
				lcd_write_char('/');
				lcd_write_hex(disp.rdaycnt);
				lcd_write_str(" (");
				lcd_write_str(dayw[disp.rwkcnt]);
				lcd_write_str(")     ");
				
				day_cnt = disp.rdaycnt;
			}
			
			g_lcd_update = 0;
		}
		
		if (g_sw8_push == 1)
		{
			//SW8: カレンダ設定を行う
			g_sw8_push = 0;
			calender_set();
		}
		
		//タイマ動作
		switch (g_timer_operation)
		{
			case TIMER_NONE:
			
				led_set(0x1);	//アラーム設定開始のSW0->LED0を点灯させる
				
				if ((push_sw_read() & 0x1) == 0x0)
				{
					//SW0が押されたのでアラーム設定モードに移行
					g_timer_operation = TIMER_VAL_SETTING;
					vr_invalid = 0;			//VRを有効化する
				}
				break;
				
			case TIMER_VAL_SETTING:
			
				led_set(0x2);	//タイマスタートのSW1->LED1を点灯させる
				
				if ((adc_val() < (current_adc_val - 100)) || (adc_val() > (current_adc_val + 100)))
				{
					//VRが回されたので（VRを無効化した時から大きく変動したので）VRの読み取りを有効とする
					vr_invalid = 0;
				}
				
				if (vr_invalid == 0)
				{
					g_timer_val = (int)((float)(4095 - adc_val()) * 1.002f) * TIMER_MAX_SEC / 4096 + 1; //ボリュームにより 1 - TIMER_MAX_SEC の範囲で設定, A/D値が4095まで達しなくても2%加算して最大値まで達する様にする
				}
				
				if (g_sw10_push == 1)
				{
					//SW10: 微調整（タイマ時間を1秒増やす）
					g_timer_val++;
					vr_invalid = 1;					//VRを無効化する
					current_adc_val = adc_val();	//現在のVRの読み取り値を保存
					g_sw10_push = 0;
					
				}
				
				if (g_sw9_push == 1)
				{
					//SW9: 微調整（タイマ時間を1秒減らす）
					g_timer_val--;
					vr_invalid = 1;					//VRを無効化する
					current_adc_val = adc_val();	//現在のVRの読み取り値を保存
					g_sw9_push = 0;
				}
				
				//g_timer_valの有効範囲
				if (g_timer_val > TIMER_MAX_SEC)
				{
					g_timer_val = TIMER_MAX_SEC;
				}
				else if (g_timer_val < 1)
				{
					g_timer_val = 1;
				}
				
				led7seg_disp_num(g_timer_val, 0);	//7セグLEDにタイマ値を表示
				if ((push_sw_read() & 0x2) == 0x0)
				{
					//SW1が押されたのでタイマスタート
					g_timer_operation = TIMER_COUNT;
					led_set(0x0);	//LEDは消灯
				}
				break;
				
			case TIMER_COUNT:
				
				if (g_timer_val > 0)
				{
					//カウント中
					led7seg_disp_num(g_timer_val, 0);	//7セグLEDにタイマ値を表示
				}
				else
				{
					//カウント終了後（アラーム鳴動中）
					led7seg_disp_num(0, 0);				//カウント終了後は0を表示
					
					if (timer_alarm == 0)
					{
						alarm_on(TIMER_ALARM_1);
						timer_alarm = TIMER_ALARM_1;
						led_set(0xff);						//LED0-LED7オン
					}
					else if ((timer_alarm == TIMER_ALARM_1) && ((g_timer_val * -1) > TIMER_ALARM_1_TIME))	//カウントが0になった後も、g_timer_val は減算が続く
					{
						alarm_on(TIMER_ALARM_2);
						timer_alarm = TIMER_ALARM_2;
					}
					
					if (push_sw_read() != 0xff)
					{
						//カウント終了後（アラーム鳴動中）に、SW0-SW7のいずれかのスイッチを押すとアラームが止まる
						alarm_off();
						led_set(0);							//LED0-LED7オフ
						led7seg_off();						//7セグLEDオフ
						g_timer_operation = TIMER_NONE;
						timer_alarm = 0;
						g_timer_val = 0;
					}
				}
				break;
		}
		
		if ((push_sw_read() & 0x80) == 0x0)
		{
			//SW7が押された場合はタイマ動作をキャンセル
			alarm_off();
			led_set(0);							//LED0-LED7オフ
			led7seg_off();						//7セグLEDオフ
			g_timer_operation = TIMER_NONE;
			timer_alarm = 0;
			g_timer_val = 0;
		}
		
		__nop();
	}
}

void alarm_on(int level)
{
	//アラームを鳴らす関数
	
	//引数
	//  level
	//    0 : 連続でピー音を鳴らす
	//    1 : ピー　ピー... ピー ピー...の繰り返し
	//    2 : ピピピピピピの繰り返し
	
	//戻り値
	//  なし
	
	int i;
	
	buzzer_freq_set(4000);	//ブザー音の高さは4,000Hzとする
	
	switch (level)
	{
		case 0:
			buzzer_on();
			break;
			
		case 1:
			for (i=0; i<10; i++) g_buzzer_pattern[i] = 0;
			g_buzzer_pattern[0] = 1;
			g_buzzer_pattern[2] = 1;
			//oxoxxxxxxxの繰り返し
			g_buzzer_flow = 1;		//パターンで鳴らす
			led8_cycle_set(10000);	//パターンの場合はdutyは関係しない
			led8_duty_set(0.02f);	//ブザーを鳴らすタイミングでLEDを一瞬点灯
			led8_on();
			break;
			
		case 2:
			g_buzzer_flow = 2;		//dutyで鳴らす
			led8_cycle_set(10000);
			led8_duty_set(0.5f);	//ブザーを鳴らすduty
			led8_on();
			break;
	}
			
}

void alarm_off(void)
{
	//アラームを止める関数
	
	//引数
	//  なし
	
	//戻り値
	//  なし
	
	buzzer_off();
	led8_off();
	
}

void calender_set(void)
{
	//カレンダ設定関数
	
	//引数
	//  なし
	
	//戻り値
	//  なし
	
	rtc_calendarcounter_value_t cal;
	int process = 0;	//0:年の変更, 1:月の変更, 2:日の変更, 3:時間の変更, 4:分の変更, 5:秒の変更, 6:本関数を抜ける
	
	int year_desision = 0;	//年の確定
	int month_desision = 0;	//月の確定
	
	int modify[6];
	int modify_max[6] = {2099, 12, 31, 23, 59, 59};
	int modify_min[6] = {2020, 1, 1, 0, 0, 0};
	
	//月ごとの日数
	unsigned char month_days[12] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};

	R_Config_RTC_Get_CalendarCounterValue(&cal);
	
	modify[0] = (cal.ryrcnt & 0x0F) + (((cal.ryrcnt & 0xF0) >> 4) * 10) + 2000;
	modify[1] = (cal.rmoncnt & 0x0F) + (((cal.rmoncnt & 0xF0) >> 4) * 10);
	modify[2] = (cal.rdaycnt & 0x0F) + (((cal.rdaycnt & 0xF0) >> 4) * 10);
	
	modify[3] = (cal.rhrcnt & 0x0F) + (((cal.rhrcnt & 0xF0) >> 4) * 10);
	modify[4] = (cal.rmincnt & 0x0F) + (((cal.rmincnt & 0xF0) >> 4) * 10);
	modify[5] = (cal.rseccnt & 0x0F) + (((cal.rseccnt & 0xF0) >> 4) * 10);
	
	calender_set_lcd_disp(process, modify);	//表示更新
	
	//SW9, SW10は押下済みの場合でもキャンセルしておく
	g_sw9_push = 0;
	g_sw10_push = 0;
	
	while (1)
	{
		if (g_sw8_push == 1)
		{
			//SW8 : 次の設定項目へ進める, 設定終了
			g_sw8_push = 0;
			process++;
			
			calender_set_lcd_disp(process, modify);	//表示更新
		}
		
		if (g_sw10_push == 1)
		{
			//SW10 : 設定項目値+1
			g_sw10_push = 0;
			modify[process] += 1;
			if (modify[process] > modify_max[process])
			{
				modify[process] = modify_min[process];
			}
			
			calender_set_lcd_disp(process, modify);	//表示更新
		}
		
		if (g_sw9_push == 1)
		{
			//SW9 : 設定項目値-1
			g_sw9_push = 0;
			modify[process] -= 1;
			if (modify[process] < modify_min[process])
			{
				modify[process] = modify_max[process];
			}
			
			calender_set_lcd_disp(process, modify);	//表示更新
		}
		
		if ((process == 1) && (year_desision == 0))
		{
			//年が確定した時点でうるう年を確定させる
			
			if ((modify[0] % 4) == 0)
			{
				//2020-2099の間には、100で割り切れる, 400で割り切れる例外なし
				month_days[1] = 29;
			}
			year_desision = 1;
		}
		
		if ((process == 2) && (month_desision == 0))
		{
			//月が確定した時点で月毎の日数を確定させる
			modify_max[2] = month_days[modify[1] - 1];
			
			month_desision = 1;
		}
		
		if (process >= 6) break;	//設定終了

	}
	
	lcd_cmd(0x0c);	//カーソルOFF
	
	cal.ryrcnt = (((modify[0] - 2000) / 10) << 4) + ((modify[0] - 2000) % 10);
	cal.rmoncnt = ((modify[1] / 10) << 4) + (modify[1] % 10);
	cal.rdaycnt = ((modify[2] / 10) << 4) + (modify[2] % 10);
	cal.rwkcnt = (unsigned char)day_of_week(modify[0], modify[1], modify[2]);
	cal.rhrcnt = ((modify[3] / 10) << 4) + (modify[3] % 10);
	cal.rmincnt = ((modify[4] / 10) << 4) + (modify[4] % 10);
	cal.rseccnt = ((modify[5] / 10) << 4) + (modify[5] % 10);
	
	//RTCに時間設定
	RTC.RSECCNT.BYTE = cal.rseccnt;
    RTC.RMINCNT.BYTE = cal.rmincnt;
    RTC.RHRCNT.BYTE = cal.rhrcnt;
    RTC.RWKCNT.BYTE = cal.rwkcnt;
    RTC.RDAYCNT.BYTE = cal.rdaycnt;
    RTC.RMONCNT.BYTE = cal.rmoncnt;
    RTC.RYRCNT.WORD = cal.ryrcnt;
}

void calender_set_lcd_disp(int process, int *modify)
{
	//カレンダ設定中のLCD表示関数
	
	//引数
	//  process : どの項目を設定中か
	//  *modify : 更新中の値
	
	//戻り値
	//  なし
	
	rtc_calendarcounter_value_t cal;

	cal.ryrcnt = (((modify[0] - 2000) / 10) << 4) + ((modify[0] - 2000) % 10);
	cal.rmoncnt = ((modify[1] / 10) << 4) + (modify[1] % 10);
	cal.rdaycnt = ((modify[2] / 10) << 4) + (modify[2] % 10);
	cal.rwkcnt = (unsigned char)day_of_week(modify[0], modify[1], modify[2]);
	cal.rhrcnt = ((modify[3] / 10) << 4) + (modify[3] % 10);
	cal.rmincnt = ((modify[4] / 10) << 4) + (modify[4] % 10);
	cal.rseccnt = ((modify[5] / 10) << 4) + (modify[5] % 10);
	
	//1行目
	//1234567890123456
	//2024/12/31 (FRY)
	lcd_hs1();
	lcd_write_str("20");
	lcd_write_hex(cal.ryrcnt);
	lcd_write_char('/');
	lcd_write_hex(cal.rmoncnt);
	lcd_write_char('/');
	lcd_write_hex(cal.rdaycnt);
	lcd_write_str(" (");
	lcd_write_str(dayw[cal.rwkcnt]);
	lcd_write_str(")     ");
	
	//2行目
	//1234567890123456
	//20:25:30
	lcd_hs2();
	lcd_write_hex(cal.rhrcnt);
	lcd_write_char(':');
	lcd_write_hex(cal.rmincnt);
	lcd_write_char(':');
	lcd_write_hex(cal.rseccnt);

	//0123456789012345
	//2024/12/20
	//17:50:31
	switch (process)
	{
		case 0:	//年の編集
			lcd_cursor_move(1, 3);
			break;
	
		case 1:	//月の編集
			lcd_cursor_move(1, 6);
			break;
			
		case 2:	//日の編集
			lcd_cursor_move(1, 9);
			break;
		
		case 3:	//時間の編集
			lcd_cursor_move(2, 1);
			break;
		
		case 4:	//分の編集
			lcd_cursor_move(2, 4);
			break;
		
		case 5:	//秒の編集
			lcd_cursor_move(2, 7);
			break;
	}
	lcd_cmd(0x0f);	//カーソルON, カーソル点滅
}

int day_of_week(int y, int m, int d)
{
	//曜日を求める関数
	
	//引数
	//  y : 年
	//  m : 月
	//  d : 日付
	
	//戻り値
	// 0 : 日曜
	// 1 : 月曜
	// ...
	// 6 : 土曜
	
	if (m < 3)
	{
		y--;
		m += 12;
	}
	
	return (365 * y + y / 4 - y / 100 + y / 400 + 306 * (m + 1) / 10 + d - 428) % 7;
}

void led7seg_off(void)
{
	//7セグLEDをオフする関数
	
	//引数
	//  なし
	
	//戻り値
	//  なし
	
	int i;
	
	for (i=1; i<=4; i++)
	{
		//各セグメントOFF
		led7seg_segument_data_set(i, 'A', 0);
		led7seg_segument_data_set(i, 'B', 0);
		led7seg_segument_data_set(i, 'C', 0);
		led7seg_segument_data_set(i, 'D', 0);
		led7seg_segument_data_set(i, 'E', 0);
		led7seg_segument_data_set(i, 'F', 0);
		led7seg_segument_data_set(i, 'G', 0);
		led7seg_segument_data_set(i, 'P', 0);
	}
}

/*----------------------------------------------------------------------
	割り込みコールバック関数
----------------------------------------------------------------------*/
void cmt3_callback_timer(void)
{
}

void rtc_callback_timer(void)
{
	//RTC割り込みのタイミングでLCD表示を更新
	
	g_lcd_update = 1;	//秒単位でLCD2行目を更新
	
	if (g_timer_operation == TIMER_COUNT)
	{
		//タイマカウント中はタイマ値を減算する
		g_timer_val--;
	}
}

void adc_callback_timer(void)
{
}

void sw8_callback_timer(void)
{
	g_sw8_push = 1;
}

void sw9_callback_timer(void)
{
	g_sw9_push = 1;
}

void sw10_callback_timer(void)
{
	g_sw10_push = 1;
}

void tpu2a_callback_timer(void)
{
	static int counter = 0;
	
	//ブザー制御
	if (g_buzzer_flow == 1)
	{
		//ピーピー..ピーピー..など特定のパターンでブザーを鳴らす
		if (g_buzzer_pattern[counter++] == 1)
		{
			buzzer_on();
			led_set(0xff);	//LED0-LED7全点灯
		}
		else
		{
			buzzer_off();
			led_set(0x0);	//LED0-LED7消灯
		}
		if (counter >= BUZZER_PATTERN_CYCLE) counter = 0;
	}
	else if (g_buzzer_flow == 2)
	{
		//ピピピピ..などの連続音でブザーを鳴らす
		buzzer_off();
		led_set(0xff);	//LED0-LED7全点灯
	}
}

void tpu2b_callback_timer(void)
{
	//ブザー制御
	if (g_buzzer_flow == 2)
	{
		buzzer_on();
	}
}
