/**
 * @file  sci.h
 * @brief SCIドライバヘッダ
 *
 * @version: 1.9 2024/10/17, $Id: $
 *
 * 本ソフトウェアの著作権は作成元である(株)北斗電子が所有するものとし、
 * (株)北斗電子は、以下の (1)-(3) の条件を満たす場合に限り、
 * 本ソフトウェア（本ソフトウェアを改変したものを含む。以下同じ）を
 * 使用・複製・改変・再配布（以下、利用と呼ぶ）することを無償で許諾する。
 *
 * (1) 本ソフトウェアをソースコードの形で利用する場合には、下記の著作
 *	   権表示、この利用条件が、そのままの形でソースコード中に含まれて
 *	   いること。
 * (2) 本ソフトウェアの一部または全てを無断で転載することを禁止するもの
 *	   とする。雑誌などへ紹介・収録の場合は(株)北斗電子に連絡願います。
 * (3) 本ソフトウェアの利用により直接的または間接的に生じるいかなる損害
 *	   からも、(株)北斗電子は一切の責任を負わないものとする。
 *
 * Copyright (C) Hokuto denshi Co,Ltd. 2021-2024
 */

#ifndef SCI_H__
#define SCI_H__

/* ユーザ側で選択する項目(*n)に関しては、readme.txtを参照してください */

/*----------------------------------------------------------------------
	マイコン種別（別ファイルで定義）(*1)
----------------------------------------------------------------------*/

//#include "../mcu_type/mcu_type.h"		//別ファイルの定義を読み込み, 別ファイルで設定しない場合はコメントアウト

/*----------------------------------------------------------------------
	マイコン種別（いずれか1つを選択）(*1')
----------------------------------------------------------------------*/

#ifndef MCU_TYPE_DEFINED	//他で定義済みの場合は、ここでの設定をスキップ

//下記5種類の内1つを有効化

#define MCU_TYPE_RX
//#define MCU_TYPE_RL78
//#define MCU_TYPE_RA
//#define MCU_TYPE_RH850
//#define MCU_TYPE_RISC_V

#endif

/*----------------------------------------------------------------------
	RL78の場合 スマートコンフィグレータ, コード生成のどちらか （どちらかを選択）(*2)
----------------------------------------------------------------------*/

#if defined(MCU_TYPE_RL78)
#define RL78_SMART_CONFIGURATOR			//スマートコンフィグレータ（RL78/G2x以降）
//#define RL78_CODE_GENERATION			//コード生成
#endif

/*----------------------------------------------------------------------
    RAの場合 SCI, SCI_Bのどちらか （どちらかを選択）(*2')
----------------------------------------------------------------------*/

#if defined(MCU_TYPE_RA)
#define RA_SCI							//通常のSCI
//#define RA_SCI_B						//SCI_B
#endif

/*----------------------------------------------------------------------
	インクルード(*3)
----------------------------------------------------------------------*/

#if defined(MCU_TYPE_RX)

#include	"r_smc_entry.h"

#elif defined(MCU_TYPE_RL78)

#ifdef RL78_SMART_CONFIGURATOR			//RL78/G23のスマートコンフィグレータでは有効化
#include	"r_smc_entry.h"	
#endif

#ifdef RL78_CODE_GENERATION				//RL78コード生成を使った場合有効化
#include	"r_cg_macrodriver.h"
#include	"r_cg_serial.h"				//※マイコン種によって
//#include	"r_cg_sau.h"				//※どちらかを選択
#endif

#elif defined(MCU_TYPE_RA)

#include	"hal_data.h"

#elif defined(MCU_TYPE_RH850)

#include	"iodefine.h"
#include	"r_cg_macrodriver.h"
#include	"Config_UART1.h"			//CHを指定

#elif defined(MCU_TYPE_RISC_V)

#include	"r_smc_entry.h"

#endif

/*----------------------------------------------------------------------
	マイコン種依存定義, SCIチャネル定義(*4)
----------------------------------------------------------------------*/

#if defined(MCU_TYPE_RX)

#define INTERRUPT_DISABLE	clrpsw_i()
#define INTERRUPT_ENABLE	setpsw_i()

#define NOP_MCU_DEP			__nop()

#define SCI_START			(void)R_Config_SCI5_Start()
#define SCI_STOP			(void)R_Config_SCI5_Stop()
#define SCI_SEND(a,b) 		(void)R_Config_SCI5_Serial_Send(a,b)
#define SCI_RECEIVE(a,b) 	(void)R_Config_SCI5_Serial_Receive(a,b)

#elif defined(MCU_TYPE_RL78)

#define INTERRUPT_DISABLE	DI()
#define INTERRUPT_ENABLE	EI()

#define NOP_MCU_DEP			__nop()

#ifdef RL78_SMART_CONFIGURATOR
#define SCI_START			(void)R_Config_UART0_Start()
#define SCI_STOP			(void)R_Config_UART0_Stop()
#define SCI_SEND(a,b) 		(void)R_Config_UART0_Send(a,b)
#define SCI_RECEIVE(a,b) 	(void)R_Config_UART0_Receive(a,b)
#endif

#ifdef RL78_CODE_GENERATION
#define SCI_START			(void)R_UART0_Start()
#define SCI_STOP			(void)R_UART0_Stop()
#define SCI_SEND(a,b) 		(void)R_UART0_Send(a,b)
#define SCI_RECEIVE(a,b) 	(void)R_UART0_Receive(a,b)
#endif

#elif defined(MCU_TYPE_RA)

#define INTERRUPT_DISABLE	__disable_irq()
#define INTERRUPT_ENABLE	__enable_irq()

#define NOP_MCU_DEP			__NOP()

#ifdef RA_SCI
#define SCI_START			(void)R_SCI_UART_Open(&g_uart9_ctrl, &g_uart9_cfg)
#define SCI_STOP			(void)R_SCI_UART_Close(&g_uart9_ctrl)
#define SCI_SEND(a,b) 		(void)R_SCI_UART_Write(&g_uart9_ctrl,a,b)
#define SCI_RECEIVE(a,b) 	(void)R_SCI_UART_Read(&g_uart9_ctrl,a,b)
#endif

#ifdef RA_SCI_B
#define SCI_START           (void)R_SCI_B_UART_Open(&g_uart9_ctrl, &g_uart9_cfg)
#define SCI_STOP            (void)R_SCI_B_UART_Close(&g_uart9_ctrl)
#define SCI_SEND(a,b)       (void)R_SCI_B_UART_Write(&g_uart9_ctrl,a,b)
#define SCI_RECEIVE(a,b)    (void)R_SCI_B_UART_Read(&g_uart9_ctrl,a,b)
#endif

#elif defined(MCU_TYPE_RH850)

#define INTERRUPT_DISABLE	DI()
#define INTERRUPT_ENABLE	EI()

#define NOP_MCU_DEP			__nop()

#define SCI_START			(void)R_Config_UART1_Start()
#define SCI_STOP			(void)R_Config_UART1_Stop()
#define SCI_SEND(a,b)		R_Config_UART1_Send(a,b)			//送信関数のみ戻り値を使用
#define SCI_RECEIVE(a,b)	(void)R_Config_UART1_Receive(a,b)

#elif defined(MCU_TYPE_RISC_V)

//RISC-Vでは全体の割り込み(mstatus.mie)を操作する事はせず、UARTの送信割り込みのみ一時的に操作する

//UART0(SAU0_ENDI0)の場合で、 SAU0_ENDI0がIESLR8 に割り付けられた場合、 R_CLIC->clicintie27_b.IE
//IESLR0 -> R_CLIC->clicintie19_b.IE
//...
//IESLR8 -> R_CLIC->clicintie27_b.IE
//使用する送信完了割り込みや使用された IESLR に応じて、"27" の部分は変更要

#define INTERRUPT_DISABLE	R_CLIC->clicintie27_b.IE=0
#define INTERRUPT_ENABLE	R_CLIC->clicintie27_b.IE=1

#define NOP_MCU_DEP			NOP()

#define SCI_START			(void)R_Config_UART0_Start()
#define SCI_STOP			(void)R_Config_UART0_Stop()
#define SCI_SEND(a,b) 		(void)R_Config_UART0_Send(a,b)
#define SCI_RECEIVE(a,b) 	(void)R_Config_UART0_Receive(a,b)

#endif

/*----------------------------------------------------------------------
	定数定義(*5)
----------------------------------------------------------------------*/
#define FLAG_SET	1
#define FLAG_CLEAR	0

#define SCI_RECEIVE_DATA_EMPTY		0xFFFF	//受信データがない場合の返却値
#define SCI_SEND_BUF_OVERFLOW		0x80	//ステータス：送信バッファが溢れた場合
#define SCI_RECEIVE_BUF_OVERFLOW	0x40	//ステータス：受信バッファが溢れた場合
#define SCI_RECEIVE_ERROR			0x08	//ステータス：受信エラーが生じた場合
#define SCI_RECEIVE_PARITY_ERROR	0x04	//ステータス：受信エラー（パリティ）が生じた場合 ※パリティ有効にしていないので発生しないはず
#define SCI_RECEIVE_FRAMING_ERROR	0x02	//ステータス：受信エラー（フレーミング）が生じた場合 ※現状RAでしかエラー種別を拾っていない
#define SCI_RECEIVE_OVERRUN_ERROR	0x01	//ステータス：受信エラー（オーバラン)が生じた場合 ※現状RAでしかエラー種別を拾っていない

#define SCI_SEND_BUF_SIZE			1024	//送信バッファ：1024 x 2ページ = 2048bytesを送信バッファとして使用
#define SCI_RECV_BUF_SIZE 			16		//受信バッファ：16bytes（キーボードからのコマンド入力を想定）
/*
RL78のG10等RAMの小さいマイコンでは、デフォルト(2kB)は、RAM容量をオーバしていますので
SCI_SEND_BUF_SIZE の値を調整してください
*/

#define SCI_USE_FLOAT						//浮動小数点の関数を使用する
/*
RL78のG10等ROMの小さいマイコンでは必要に応じてコメントアウトしてください
（RL78では、コメントアウトする事で4kB程度ROM消費量が減ります）
*/

#define SCI_USE_TX							//送信機能を使用する
#define SCI_USE_RX							//受信機能を使用する
/*
送信と受信の片方しか使用しない場合は、使用しない側をコメントアウトしてください
*/

/*----------------------------------------------------------------------
	外部参照（関数）
----------------------------------------------------------------------*/
extern void sci_start(void);
extern void sci_stop(void);
#ifdef SCI_USE_TX
extern void sci_write_char(unsigned char c);
extern void sci_write_uint8_hex(unsigned char c);
extern void sci_write_uint16_hex(unsigned short s);
extern void sci_write_uint32_hex(unsigned long l);
extern void sci_write_uint8(unsigned char num);
extern void sci_write_uint16(unsigned short num);
extern void sci_write_uint32(unsigned long num);
extern void sci_write_int8(char num);
extern void sci_write_int16(short num);
extern void sci_write_int32(long num);
extern void sci_write_str(const char *str);
extern void sci_write_flush(void);
#endif
#ifdef SCI_USE_RX
extern unsigned short sci_read_char(unsigned char *c);
extern unsigned short sci_read_str(char *str, unsigned short size);
extern unsigned short sci_read_data_size(void);
extern void sci_read_buf_clear(void);
#endif
#ifdef SCI_USE_FLOAT
extern int float2str(float value, int num, char *str);
extern int float2str_eformat(float value, int num, char *str);
extern int double2str(double value, int num, char *str);
extern int double2str_eformat(double value, int num, char *str);
#endif

/*----------------------------------------------------------------------
	割り込み関数から呼ばれる関数
----------------------------------------------------------------------*/
#ifdef SCI_USE_TX
extern void intr_sci_send_end(void);
#endif
#ifdef SCI_USE_RX
extern void intr_sci_receive_end(void);
extern void intr_sci_receive_error(void);
#endif

/*
 RX, RL78, RH850, RISC_Vの場合、ツールで生成した コード( Config_????_user.c / r_cg_????_user.c )
 内に以下の記載を追加 してください（詳しくは readme.txt を参照）

・Includesの「/ * Start user code for include.」と「/ * End user code.」の間に

#include "sci.h"

 を記載する。

・送信完了時に呼び出される( ~transmitend(), ~sendend() )内に記載（送信機能使用時）

intr_sci_send_end();

・受信完了時に呼び出される( ~receiveend() )内に記載（受信機能使用時）

intr_sci_receive_end();

・受信エラー時に呼び出される( ~receiveerror(), ~error() )内に記載（受信機能使用時）

intr_sci_receive_error();

*/

/*----------------------------------------------------------------------
	外部参照（変数）
----------------------------------------------------------------------*/
#ifdef SCI_USE_TX
extern unsigned char g_sci_send_buf[2][SCI_SEND_BUF_SIZE];
#endif
#ifdef SCI_USE_RX
extern unsigned char g_sci_recv_buf[SCI_RECV_BUF_SIZE];
#endif
#ifdef SCI_USE_TX
extern volatile unsigned short g_sci_send_buf_index[2];
#endif
#ifdef SCI_USE_TX
extern volatile unsigned char g_sci_send_buf_page_index; 
extern volatile unsigned char g_sci_send_flag;
extern volatile unsigned char g_sci_send_nowait_flag;
#endif
#ifdef SCI_USE_RX
extern volatile unsigned short g_sci_recv_buf_index1 ;
extern volatile unsigned short g_sci_recv_buf_index2 ;
extern unsigned char g_sci_recv_data;
#endif
extern volatile unsigned char g_sci_status;
extern volatile unsigned char g_sci_start;

#endif
/* End of File */
